package com.xebialabs.deployit.task.archive

import javax.jcr.Node

import com.google.common.base.{Preconditions, Strings}
import com.xebialabs.deployit.engine.api.execution.TaskWithBlock
import com.xebialabs.deployit.jcr.JcrUtils._
import com.xebialabs.deployit.task.TaskType
import com.xebialabs.deployit.task.TaskMetadata._
import NodeNames.{INSTANTIATE_CONTROL_TASK_NAME, DESTROY_CONTROL_TASK_NAME}


class CloudTaskArchiver(val task: TaskWithBlock) extends TaskArchiver {

  override def save(rootNode: Node) {
    val taskNode = createTaskNode(rootNode)

    fillTaskNode(taskNode)

    val operation = getMetadata(task, CLOUD_OPERATION)
    val environmentId = EnvironmentId.encode(getMetadata(task, CLOUD_ENVIRONMENT_ID))
    val environmentTemplateId = ConfigurationId.encode(getMetadata(task, CLOUD_ENVIRONMENT_TEMPLATE_ID))

    taskNode.setProperty(CLOUD_OPERATION, operation)
    taskNode.setProperty(CLOUD_ENVIRONMENT_ID, environmentId)
    taskNode.setProperty(CLOUD_ENVIRONMENT_TEMPLATE_ID, environmentTemplateId)
  }

}

class CloudTaskReader(val taskNode: Node) extends TaskReader {
  override def readWithoutStep(): ArchivedTask = {
    val task = taskWithoutSteps

    putMetadata(task, CLOUD_OPERATION, getProperty(taskNode, CLOUD_OPERATION))
    putMetadata(task, CLOUD_ENVIRONMENT_TEMPLATE_ID, ConfigurationId.decode(getProperty(taskNode, CLOUD_ENVIRONMENT_TEMPLATE_ID)))
    putMetadata(task, CLOUD_ENVIRONMENT_ID, EnvironmentId.decode(getProperty(taskNode, CLOUD_ENVIRONMENT_ID)))

    task
  }
}

object CloudTask {

  def unapply(task: TaskWithBlock): Option[TaskArchiver] = {
    lazy val operation = getMetadata(task, CLOUD_OPERATION)

    if (TaskType.CONTROL == TaskType.valueOf(getMetadata(task, TASK_TYPE)) &&
      metadataContainsKey(task, CLOUD_OPERATION) &&
      ((INSTANTIATE_CONTROL_TASK_NAME == operation) || DESTROY_CONTROL_TASK_NAME == operation)) {

      Preconditions.checkState(!Strings.nullToEmpty(operation).isEmpty, CLOUD_OPERATION + " in provisioning task must be set", "")
      Preconditions.checkState((operation == INSTANTIATE_CONTROL_TASK_NAME) || (operation == DESTROY_CONTROL_TASK_NAME), CLOUD_OPERATION + " in provisioning task must be [instantiate|destroy]", "")
      Preconditions.checkState(!Strings.nullToEmpty(getMetadata(task, CLOUD_ENVIRONMENT_ID)).isEmpty, CLOUD_ENVIRONMENT_ID + " in provisioning task must be set", "")
      Preconditions.checkState(!Strings.nullToEmpty(getMetadata(task, CLOUD_ENVIRONMENT_TEMPLATE_ID)).isEmpty, CLOUD_ENVIRONMENT_TEMPLATE_ID + " in provisioning task must be set", "")

      Some(new CloudTaskArchiver(task))
    } else {
      None
    }
  }

  def unapply(taskNode: Node): Option[TaskReader] = {

    if (taskNode.hasProperty(CLOUD_OPERATION)) {
      Some(new CloudTaskReader(taskNode))
    } else None
  }

}
