package com.xebialabs.deployit.service.importer.source;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import org.apache.commons.httpclient.*;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.io.Files;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.util.GuavaFiles;

import static com.google.common.base.Strings.emptyToNull;

public class UrlSource implements ImportSource {

    private static final Logger logger = LoggerFactory.getLogger(UrlSource.class);

    private static final String CONTENT_DISPOSITION = "Content-disposition";
    private static final String ATTACHMENT = "attachment";
    private static final String FILENAME = "filename";

    private final URL location;
    private final String user;
    private final String password;
    private FileSource downloaded;
    private File tempDir;

    public UrlSource(URL location, String user, String password) {
        this.location = location;
        this.user = user;
        this.password = password;
    }

    @Override
    public File getFile() {
        if (downloaded == null) {
            download();
        }

        return downloaded.getFile();
    }

    @Override
    public void cleanUp() {
        if (downloaded != null) {
            downloaded.cleanUp();
        }
        if (tempDir != null) {
            GuavaFiles.deleteQuietly(tempDir);
        }
    }

    private void download() {
        try {
            logger.debug("Preparing to download package from {}", location);
            HttpMethod method = executeGetRequest();
            saveFile(method);
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException("Invalid URL", e);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    protected HttpMethod executeGetRequest() throws URISyntaxException, IOException {
        URI uri = location.toURI();
        HttpClient client = new HttpClient();
        HttpMethod method = new GetMethod(uri.toString());
        configureAuthentication(client);
        int statusCode = client.executeMethod(method);
        if (statusCode != HttpStatus.SC_OK) {
            throw new RuntimeIOException("Failed to download package,status="+ statusCode +", from url " + location);
        }
        return method;
    }

    private void configureAuthentication(HttpClient client) {
        if (user != null) {
            client.getParams().setAuthenticationPreemptive(true);
            Credentials credentials = new UsernamePasswordCredentials(user, password);
            client.getState().setCredentials(AuthScope.ANY, credentials);
        }
    }

    private void saveFile(HttpMethod method) throws IOException {
        try(InputStream in = method.getResponseBodyAsStream()) {
            tempDir = Files.createTempDir();
            File archive = new File(tempDir, getFileName(method));
            Files.asByteSink(archive).writeFrom(in);
            this.downloaded = new FileSource(archive, true);
            logger.debug("Successfully downloaded file {}", downloaded.getFile().getName());
        }
    }

    private String getFileName(HttpMethod method) throws URIException {
        String filename = getFilenameFromHeader(method);
        if (filename == null) {
            logger.trace("File name could not be resolved through 'Content-disposition' header");
            filename = getFileNameFromPath(method.getURI().getPath());
        }
        logger.debug("File name resolved as {}", filename);
        return filename;
    }

    private String getFilenameFromHeader(HttpMethod method) {
        // check the Content-disposition header for a "filename" param
        Header contentDisposition = method.getResponseHeader(CONTENT_DISPOSITION);
        if (contentDisposition != null) {
            for (HeaderElement element : contentDisposition.getElements()) {
                if (element.getName().equalsIgnoreCase(ATTACHMENT)) {
                    NameValuePair filenameParam = element.getParameterByName(FILENAME);
                    return (filenameParam != null) ? emptyToNull(filenameParam.getValue()) : null;
                }
            }
        }
        return null;
    }

    private String getFileNameFromPath(final String path) {
        logger.trace("Getting target file name from path {}", path);
        return new File(path).getName();
    }

    @Override
    public String toString() {
        return "UrlSource[" + location + "]";
    }
}
