package com.xebialabs.deployit.service.deployment;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployedContainer;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.IncorrectArgumentException;
import static java.lang.String.format;

@Component
public class TypeCalculator {

    public Type findMostSpecificDeployedType(final Type deployableType, final Type containerType) {
        List<Type> deployedTypesForDeployableAndContainer = findDeployedTypesForDeployableAndContainerTypes(deployableType, containerType);
        List<Type> copy = filterMostSpecific(deployedTypesForDeployableAndContainer);

        if (copy.isEmpty()) {
            return null;
        }

        if (copy.size() == 1) {
            return copy.get(0);
        }

        throw new IncorrectArgumentException(format("Found %s for %s and %s, expected exactly 1. Before filtering we had: %s", copy, deployableType, containerType, deployedTypesForDeployableAndContainer));
    }

    public List<Type> findAllMostSpecificDeployedTypesForDeployableAndContainerTypes(final Type deployableType, final Type containerType) {
        return filterMostSpecific(findDeployedTypesForDeployableAndContainerTypes(deployableType, containerType));
    }

    List<Type> filterMostSpecific(final List<Type> deployedTypesForDeployableAndContainer) {
        List<Type> copy = newArrayList(deployedTypesForDeployableAndContainer);

        for (Type currentType : deployedTypesForDeployableAndContainer) {
            Descriptor currentDescriptor = DescriptorRegistry.getDescriptor(currentType);
            Type currentDeployableType = currentDescriptor.getDeployableType();
            Type currentContainerType = currentDescriptor.getContainerType();

            for (Iterator<Type> copyIt = copy.iterator(); copyIt.hasNext();) {
                Type comparedType = copyIt.next();
                Descriptor comparedDescriptor = DescriptorRegistry.getDescriptor(comparedType);
                Type comparedDeployableType = comparedDescriptor.getDeployableType();
                Type comparedContainerType = comparedDescriptor.getContainerType();

                boolean deployableEquals = comparedDeployableType.equals(currentDeployableType);
                boolean containerEquals = comparedContainerType.equals(currentContainerType);
                boolean containerSupertype = comparedContainerType.isSuperTypeOf(currentContainerType);
                boolean deployableSupertype = comparedDeployableType.isSuperTypeOf(currentDeployableType);

                boolean isSupertype = comparedType.isSuperTypeOf(currentType);

                String currentTypeString = format("%s<%s, %s>", currentType, currentDeployableType, currentContainerType);
                String comparedTypeString = format("%s<%s, %s>", comparedType, comparedDeployableType, comparedContainerType);
                if (containerSupertype && deployableEquals) {
                    logger.debug("Filtering out [{}] because is has a container supertype wrt. [{}]", comparedTypeString, currentTypeString);
                    copyIt.remove();
                } else if (deployableSupertype && containerEquals) {
                    logger.debug("Filtering out [{}] because is has a deployable supertype wrt. [{}]", comparedTypeString, currentTypeString);
                    copyIt.remove();
                } else if (deployableSupertype && containerSupertype) {
                    logger.debug("Filtering out [{}] because is has a container and deployable supertype wrt. [{}]", comparedTypeString, currentTypeString);
                    copyIt.remove();
                } else if (isSupertype) {
                    logger.debug("Filtering out [{}] because is a supertype of [{}]", comparedTypeString, currentTypeString);
                    copyIt.remove();
                }
            }
        }
        logger.debug("Result of filtering most specific: {}", copy);
        return copy;
    }

    public List<Type> findDeployedTypesForDeployableAndContainerTypes(final Type deployableType, final Type containerType) {
        Collection<Type> deployedTypes = DescriptorRegistry.getSubtypes(Type.valueOf(EmbeddedDeployedContainer.class));
        final Descriptor deployableDesc = DescriptorRegistry.getDescriptor(deployableType);
        final Descriptor containerDesc = DescriptorRegistry.getDescriptor(containerType);
        ArrayList<Type> result = newArrayList(Collections2.filter(deployedTypes, new Predicate<Type>() {
            @Override
            public boolean apply(Type input) {
                Descriptor deployedDesc = DescriptorRegistry.getDescriptor(input);
                return !deployedDesc.isVirtual()
                        && deployableDesc.isAssignableTo(deployedDesc.getDeployableType())
                        && containerDesc.isAssignableTo(deployedDesc.getContainerType());
            }
        }));
        logger.debug("Found types {} for <{}, {}>", new Object[] {result, deployableType, containerType});
        return result;
    }

    private static final Logger logger = LoggerFactory.getLogger(TypeCalculator.class);
}
