package com.xebialabs.deployit.service.deployment

import java.io.File

import com.xebialabs.deployit.engine.api.execution.TaskExecutionState.{ABORTED, EXECUTED, FAILED, STOPPED}
import com.xebialabs.deployit.engine.tasker.{IEngine, Task, TaskExecutionContext, TaskSpecification}
import com.xebialabs.deployit.repository.WorkDir
import com.xebialabs.deployit.task.TaskMetadata._
import com.xebialabs.deployit.task.{TaskType, WorkdirCleanerTrigger}
import com.xebialabs.overthere.local.LocalFile
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.convert.wrapAll._

@Component
class RollbackService @Autowired() (engine: IEngine, deploymentService: DeploymentService) extends Logging {
  def rollback(task: Task): TaskSpecification = {
    logger.debug(s"Going to rollback task ${task.getId()}")
    val context: TaskExecutionContext = task.getContext
    val workdirCleanerTrigger: WorkdirCleanerTrigger = getAndUnregisterWorkdirCleaner(context)
    val partialCommitTrigger: ImprovedPartialCommitTrigger = getPartialCommitTrigger(context)
    val rollbackSpecification: TaskSpecification = createRollbackSpecification(task, partialCommitTrigger, workdirCleanerTrigger.getWorkDirs.toList)
    archive(task)
    putMetadata(rollbackSpecification, TASK_TYPE, TaskType.ROLLBACK.name)
    putMetadata(rollbackSpecification, ROLLBACK_TASK, task.getId())
    rollbackSpecification
  }

  private def createRollbackSpecification(task: Task, partialCommitTrigger: ImprovedPartialCommitTrigger, workDirs: List[WorkDir]): TaskSpecification = {
    workDirs.foreach { wd =>
      val from: LocalFile = LocalFile.from(new File(wd.getPath))
      if (!from.exists()) {
        debug(s"Re-creating workdir ${from.getPath} because it was deleted by the original deployment.")
        from.mkdir()
      }
    }

    val rollbackSpec = partialCommitTrigger.checkpointManager.prepareRollback()
    deploymentService.getTaskFullSpecification(rollbackSpec, task.getWorkDir, workDirs:_*)
  }

  private def getAndUnregisterWorkdirCleaner(context: TaskExecutionContext): WorkdirCleanerTrigger = {
    val name: String = classOf[WorkdirCleanerTrigger].getName
    context.unsetAttribute(name).get.asInstanceOf[WorkdirCleanerTrigger]
  }

  private def getPartialCommitTrigger(context: TaskExecutionContext): ImprovedPartialCommitTrigger = {
    val name: String = classOf[ImprovedPartialCommitTrigger].getName
    context.getAttribute(name).asInstanceOf[ImprovedPartialCommitTrigger]
  }

  private def archive(task: Task): Unit = {
    if (task.getState == EXECUTED) {
      engine.archive(task.getId())
    } else if (Set(STOPPED, ABORTED, FAILED).contains(task.getState())) {
      engine.cancel(task.getId())
    } else {
      throw new IllegalStateException(s"Can only rollback a STOPPED, FAILED, ABORTED or EXECUTED task [${task.getId()} (${task.getState()})]")
    }
  }
}
