package com.xebialabs.deployit.service.deployment;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.*;
import com.xebialabs.deployit.deployment.planner.*;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskPackageDependency;
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.engine.tasker.PhaseContainer;
import com.xebialabs.deployit.engine.tasker.TaskSpecification;
import com.xebialabs.deployit.engine.tasker.TaskStep;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.PreviewStep;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployed;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.xld.AppliedDistribution;
import com.xebialabs.deployit.repository.RepositoryAdapterFactory;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.task.TaskType;
import com.xebialabs.deployit.task.WorkdirCleanerTrigger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Nullable;
import java.io.File;
import java.io.StringWriter;
import java.util.*;
import java.util.stream.Collectors;

import static com.google.common.base.Predicates.instanceOf;
import static com.google.common.base.Predicates.or;
import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.task.TaskMetadata.*;
import static com.xebialabs.deployit.task.TaskType.*;

@Component("deploymentService")
public class DeploymentService {

    private final MultiDeploymentPlanner planner;
    private RepositoryAdapterFactory repositoryFactory;

    @Autowired
    public DeploymentService(MultiDeploymentPlanner planner, RepositoryAdapterFactory repositoryFactory) {
        this.planner = planner;
        this.repositoryFactory = repositoryFactory;
    }

    public MultiDeltaSpecification prepareInitialSpecification(final Deployment deployment, final Map<String, DeployedApplication> existingDepoyedApplications) {
        return MultiDeltaSpecification.withDependencies(
                prepareInitialSpecification(deployment),
                prepareDependencyDeployments(deployment.getRequiredDeployments(), existingDepoyedApplications));
    }

    private DeltaSpecification prepareInitialSpecification(final Deployment deployment) {
        return prepareInitialSpecification(extractDeployedApplicationWithDeployeds(deployment));
    }

    DeltaSpecification prepareInitialSpecification(final AppliedDistribution deployedApplication) {
        DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().initial(deployedApplication);
        DeploymentOperationCalculator.calculate(builder, Sets.<Deployed>newHashSet(), deployedApplication.getDeployeds());
        return builder.build();
    }

    public MultiDeltaSpecification prepareUpgradeSpecification(final Deployment deployment, final Map<String, DeployedApplication> existingDepoyedApplications) {
        return MultiDeltaSpecification.withDependencies(
                prepareUpgradeSpecification(deployment, existingDepoyedApplications.get(deployment.getDeployedApplication().getId())),
                prepareDependencyDeployments(deployment.getRequiredDeployments(), existingDepoyedApplications));
    }

    private DeltaSpecification prepareUpgradeSpecification(final Deployment deployment, final DeployedApplication existingDeployedApplication) {
        return prepareUpgradeSpecification(extractDeployedApplicationWithDeployeds(deployment), existingDeployedApplication);
    }

    DeltaSpecification prepareUpgradeSpecification(final AppliedDistribution newDeployedApplication, final DeployedApplication existingDeployedApplication) {
        // The existingDeployedApplication does not have any 'transient' properties set. These properties are used for setting
        // deployment behaviour, and in case of rollback are taken from the now 'existing' deployed application.
        copyTransientProperties(newDeployedApplication, existingDeployedApplication);
        final DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().upgrade(existingDeployedApplication, newDeployedApplication);

        logger.trace("Incoming Deployeds {}", newDeployedApplication.getDeployeds());
        logger.trace("Existing Deployeds {}", existingDeployedApplication.getDeployeds());

        DeploymentOperationCalculator.calculate(builder, existingDeployedApplication.getDeployeds(), newDeployedApplication.getDeployeds());

        return builder.build();
    }

    private List<DeltaSpecification> prepareDependencyDeployments(List<Deployment> requiredDeployments, final Map<String, DeployedApplication> existingDepoyedApplications) {
        return Lists.newArrayList(Iterables.transform(requiredDeployments, input -> prepareDependencyDeployment(input, existingDepoyedApplications)));
    }

    private DeltaSpecification prepareDependencyDeployment(Deployment requiredDeployment, final Map<String, DeployedApplication> existingDepoyedApplications) {
        switch (requiredDeployment.getDeploymentType()) {
            case INITIAL:
                return prepareInitialSpecification(requiredDeployment);
            case UPDATE:
                return prepareUpgradeSpecification(requiredDeployment, existingDepoyedApplications.get(requiredDeployment.getDeployedApplication().getId()));
            default:
                throw new IllegalArgumentException("Deployment " + requiredDeployment.getId() + " has invalid type " + requiredDeployment.getDeploymentType());
        }
    }

    private static AppliedDistribution extractDeployedApplicationWithDeployeds(Deployment deployment) {
        AppliedDistribution deployedApp = (AppliedDistribution) deployment.getDeployedApplication();
        deployedApp.addDeployeds(getDeployeds(deployment));
        return deployedApp;
    }

    @SuppressWarnings("rawtypes")
    private static HashSet<Deployed> getDeployeds(Deployment deployment) {
        Collection<ConfigurationItem> deployedEntities = deployment.getDeployeds();
        FluentIterable<ConfigurationItem> from = from(deployedEntities);
        checkArgument(from.allMatch(or(instanceOf(Deployed.class), instanceOf(EmbeddedDeployed.class))), "The Deployment can only contain Deployed or EmbeddedDeployed configuration items");
        return Sets.newHashSet(from.filter(Deployed.class));
    }

    private void copyTransientProperties(AppliedDistribution newDeployment, DeployedApplication existingDeployment) {
        Descriptor descriptor = newDeployment.getType().getDescriptor();
        logger.debug("Copying transient properties to previous deployed application for {}", newDeployment.getId());
        for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
            if (!pd.isTransient()) {
                logger.trace("Skipping copy of non-transient property {}", pd.getFqn());
                continue;
            }

            if (pd.get(newDeployment) == null || pd.get(existingDeployment) != null) {
                logger.trace("Skipping copy of property {} as it is either not set on 'new', or already set on 'existing'", pd.getFqn());
                continue;
            }

            logger.debug("Copying transient property {} to existing deployment", pd.getFqn());
            pd.set(existingDeployment, pd.get(newDeployment));
        }
    }

    @SuppressWarnings("rawtypes")
    public MultiDeltaSpecification prepareUndeployment(final AppliedDistribution deployedApplication) {
        DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().undeploy(deployedApplication);
        DeploymentOperationCalculator.calculate(builder, deployedApplication.getDeployeds(), Sets.<Deployed>newHashSet());
        return new MultiDeltaSpecification(builder.build());
    }

    TaskSpecification getTaskSpecification(DeltaSpecification deltaSpecification, WorkDir currentWorkDir, WorkDir... workdirsToCleanup) {
        return getTaskFullSpecification(new MultiDeltaSpecification(deltaSpecification), currentWorkDir, workdirsToCleanup);
    }

    public TaskSpecification getTaskFullSpecification(MultiDeltaSpecification spec, WorkDir currentWorkDir, WorkDir... workdirsToCleanup) {
        TaskSpecification taskSpec = null;
        switch (spec.getMainOperation()) {
            case CREATE:
                taskSpec = getTaskSpecification("Initial deployment of " + spec.getMainDeployedApplication().getId(), spec, currentWorkDir, workdirsToCleanup);
                addMetadata(taskSpec, spec.getMainDeployedApplication(), INITIAL);
                break;
            case DESTROY:
                taskSpec = getTaskSpecification("Undeployment of " + spec.getMainPreviousDeployedApplication().getId(), spec, currentWorkDir, workdirsToCleanup);
                addMetadata(taskSpec, spec.getMainPreviousDeployedApplication(), UNDEPLOY);
                break;
            case MODIFY:
                String description = "Update deployment of " + spec.getMainPreviousDeployedApplication().getId();
                taskSpec = getTaskSpecification(description, spec, currentWorkDir, workdirsToCleanup);
                addMetadata(taskSpec, spec.getMainDeployedApplication(), UPGRADE);
                break;
            case NOOP:
                break;
        }

        return taskSpec;
    }

    private TaskSpecification getTaskSpecification(final String description, final MultiDeltaSpecification specification, WorkDir currentWorkDir, WorkDir... workdirsToCleanup) {
        unsetTokensForRealityPush(specification);
        PhasedPlan plan = planner.plan(specification, repositoryFactory.create(new File(currentWorkDir.getPath())));

        String taskId = UUID.randomUUID().toString();
        logger.info("Generated plan for task {}:\n{}", taskId, plan.writePlan(new StringWriter()));

        PhaseContainer executionBlock = Plans.toBlockBuilder(plan).build();

        TaskSpecification spec = new TaskSpecification(taskId, description, Permissions.getAuthentication(), currentWorkDir, executionBlock, null, true, true);
        spec.getListeners().addAll(plan.getListeners());
        ArrayList<WorkDir> workDirs = newArrayList(workdirsToCleanup);
        workDirs.add(currentWorkDir);
        spec.getListeners().add(new WorkdirCleanerTrigger(workDirs));
        spec.getListeners().add(new ReferentialIntegrityTrigger(specification));
        spec.getListeners().add(createPartialCommitTrigger(specification, plan));

        addTaskRefs(specification, spec);

        return spec;
    }

    private void addTaskRefs(final MultiDeltaSpecification specification, final TaskSpecification spec) {
        Iterable<DeltaSpecification> dependantSpecs = specification.getDeltaSpecifications().stream().filter(input -> !specification.getMainDeltaSpecification().equals(input)).collect(Collectors.toList());

        Iterable<TaskPackageDependency> dependencies = Iterables.transform(dependantSpecs, new Function<DeltaSpecification, TaskPackageDependency>() {
            AppliedDistribution deployedApplication(DeltaSpecification spec) {
                if (spec.getAppliedDistribution() != null) {
                    return spec.getAppliedDistribution();
                } else {
                    return spec.getPreviousAppliedDistribution();
                }
            }

            @Nullable
            @Override
            public TaskPackageDependency apply(DeltaSpecification input) {
                AppliedDistribution app = deployedApplication(input);
                return new TaskPackageDependency(app.getName(), app.getVersion().getVersion());
            }
        });

        for (TaskPackageDependency dependency : dependencies) {
            spec.getPackageDependencies().add(dependency);
        }
    }

    protected ExecutionStateListener createPartialCommitTrigger(MultiDeltaSpecification specification, Plan plan) {
        return new ImprovedPartialCommitTrigger(specification, plan.findCheckpoints());
    }

    private static void unsetTokensForRealityPush(final MultiDeltaSpecification fullSpec) {
        for (AppliedDistribution deployedApplication : fullSpec.getAllDeployedApplications()) {
            deployedApplication.set$token(null);
        }

        for (Delta delta : fullSpec.getAllDeltas()) {
            ConfigurationItem ci = delta.getDeployed();
            unsetTokenOnCi(ci);
        }
    }

    private static void unsetTokenOnCi(ConfigurationItem ci) {
        if (ci instanceof BaseConfigurationItem) {
            BaseConfigurationItem bci = (BaseConfigurationItem) ci;
            bci.set$token(null);
            unsetTokensOnEmbeddeds(bci);
        }
    }

    private static void unsetTokensOnEmbeddeds(BaseConfigurationItem deployed) {
        Collection<PropertyDescriptor> propertyDescriptors = deployed.getType().getDescriptor().getPropertyDescriptors();
        Collection<PropertyDescriptor> embeddedProperties = Collections2.filter(propertyDescriptors, new Predicate<PropertyDescriptor>() {
            @Override
            public boolean apply(PropertyDescriptor input) {
                return input.isAsContainment()
                        && EnumSet.of(PropertyKind.LIST_OF_CI, PropertyKind.SET_OF_CI).contains(input.getKind())
                        && input.getReferencedType().instanceOf(Type.valueOf(EmbeddedDeployed.class));
            }
        });
        for (PropertyDescriptor embeddedProperty : embeddedProperties) {
            Collection<ConfigurationItem> c = (Collection<ConfigurationItem>) embeddedProperty.get(deployed);
            for (ConfigurationItem configurationItem : c) {
                unsetTokenOnCi(configurationItem);
            }
        }

    }

    private static void addMetadata(TaskSpecification spec, AppliedDistribution deployedApp, TaskType taskType) {
        putMetadata(spec, ENVIRONMENT_ID, deployedApp.getEnvironment().getId());
        putMetadata(spec, ENVIRONMENT, deployedApp.getEnvironment().getName());
        putMetadata(spec, VERSION, deployedApp.getVersion().getName());
        putMetadata(spec, APPLICATION, deployedApp.getVersion().getDistribution().getName());
        putMetadata(spec, TASK_TYPE, taskType.name());
    }

    private static List<StepState> asTaskSteps(StepPlan plan) {
        return newArrayList(transform(plan.getStepsWithPlanningInfo(), (Function<StepPlan.StepWithPlanningInfo, StepState>) input -> {
            TaskStep taskStep = new TaskStep(input.getStep());
            int i = 0;
            taskStep.getMetadata().put("order", Integer.toString(input.getStep().getOrder()));
            taskStep.getMetadata().put("previewAvailable", Boolean.toString(input.getStep() instanceof PreviewStep));
            taskStep.getMetadata().put("rule", input.getRule());
            for (Delta delta : input.getDeltas()) {
                taskStep.getMetadata().put("deployed_" + i++, getActiveDeployed(delta).getId());
            }
            return taskStep;
        }));
    }

    private static Deployed getActiveDeployed(Delta delta) {
        return delta.getOperation() == Operation.DESTROY ? delta.getPrevious() : delta.getDeployed();
    }

    private static final Logger logger = LoggerFactory.getLogger(DeploymentService.class);
}
