package com.xebialabs.deployit.core.upgrade

import com.xebialabs.deployit.listener.version.VersionOrdering
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry.getSubtypes
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.JcrPathHelper
import com.xebialabs.deployit.server.api.repository.RawRepository
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, UpgradeException, Version}
import grizzled.slf4j.Logging

import scala.collection.convert.wrapAll._
import scala.collection.mutable
import scala.util.{Failure, Success, Try}

class Deployit500LastVersionUpgrader extends Upgrade with Logging {

  import com.xebialabs.deployit.listener.version.IdExtensions._

  val LAST_VERSION_PROPERTY_NAME = "lastVersion"

  def doUpgrade(repository: RawRepository): Boolean = Try {
    logger.info("*** Running Deployit 5.0.0 Upgrade: Last Version Number ***");

    val lastVersions = mutable.Map[String, String]()
    val versionType: Type = Type.valueOf("udm.Version")

    (getSubtypes(versionType).toList :+ versionType).foreach { t =>
      logger.info(s"Searching all nodes of type $t")
      val nodes = repository.findNodesByType(t)
      nodes.foreach { node =>
        val versionId = JcrPathHelper.getIdFromAbsolutePath(node.getPath)
        val applicationId = versionId.getParent
        val currentVersion = versionId.getName

        lastVersions(applicationId) = lastVersions.get(applicationId) match {
          case Some(lastVersion) => VersionOrdering.max(lastVersion, currentVersion)
          case None => currentVersion
        }
      }
    }

    repository.findNodesByType(Type.valueOf("udm.Application")).foreach { applicationNode =>
      val applicationId = JcrPathHelper.getIdFromAbsolutePath(applicationNode.getPath)
      val lastVersion = lastVersions.getOrElse(applicationId, VersionOrdering.EMPTY_LAST_VERSION)

      logger.info(s"Update last version [$applicationId] - $lastVersion")

      applicationNode.setProperty(LAST_VERSION_PROPERTY_NAME, lastVersion)
    }

    logger.info("*** Done Deployit 5.0.0 upgrade: Last Version Number ***");
  } match {
    case Failure(e) => throw new UpgradeException(s"Failed to run ${this.getClass.getSimpleName}.", e.asInstanceOf[Exception])
    case Success(_) => true
  }

  def upgradeVersion(): Version = Version.valueOf("deployit", "5.0.0")
}
