package com.xebialabs.deployit.service.deployment;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.xld.AppliedDistribution;
import com.xebialabs.deployit.plugin.api.xld.Domain;
import com.xebialabs.deployit.plugin.api.xld.DistributionVersion;
import com.xebialabs.deployit.service.replacement.Dictionaries;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.service.replacement.Dictionaries.of;
import static java.util.stream.Collectors.toList;

@Component
public class DeployedService {

    private final DeployedApplicationFactory deployedApplicationFactory;
    private final DeployedProcessorsFactory deployedProcessorsFactory;

    @Autowired
    public DeployedService(DeployedApplicationFactory deployedApplicationFactory, DeployedProcessorsFactory deployedProcessorsFactory) {
        this.deployedApplicationFactory = deployedApplicationFactory;
        this.deployedProcessorsFactory = deployedProcessorsFactory;
    }

    public AppliedDistribution generateDeployedApplication(Type deployedApplicationType, DistributionVersion version, Domain env, Set<String> missingPlaceholdersAggregator, Map<String, String> userProvidedPlaceholder) {
        Dictionaries dictionaries = of(env.getDictionaries()).filterBy(version.getDistribution()).withAdditionalEntries(userProvidedPlaceholder);
        return deployedApplicationFactory.createDeployedApplication(deployedApplicationType, version, env, dictionaries.consolidate(), missingPlaceholdersAggregator);
    }

    public GeneratedDeployeds generateSelectedDeployeds(AppliedDistribution appliedDistribution, List<ConfigurationItem> deployableCis, DistributionVersion version, Domain env) {
        return generateSelectedDeployeds(appliedDistribution, deployableCis, env.getMembers(), of(env.getDictionaries()).filterBy(version.getDistribution()));
    }

    public GeneratedDeployeds generateSelectedDeployeds(AppliedDistribution appliedDistribution, List<ConfigurationItem> deployableCis, Set<? extends Container> containers, Dictionaries dictionaries) {
        List<Deployable> deployables = Lists.transform(deployableCis, from -> {
            checkArgument(from instanceof Deployable, "The entity %s is not a deployable", from.getId());
            return (Deployable) from;
        });

        return generateDeployedsOfType(appliedDistribution, deployables, containers, null, dictionaries);
    }

    public GeneratedDeployeds createSelectedDeployed(AppliedDistribution appliedDistribution, Deployable deployable, Container container, Type deployedType, Version version, Environment env) {
        logger.debug("Creating deployed for [{}] and [{}]", deployable, container);

        Dictionaries dictionaries = of(env.getDictionaries()).filterBy(version.getApplication());
        return generateDeployedsOfType(appliedDistribution, newArrayList(deployable), newArrayList(container), deployedType, dictionaries, true);
    }

    public GeneratedDeployeds generateUpgradedDeployeds(AppliedDistribution appliedDistribution, Version newPackage, DeployedApplication previousDeployedApplication) {
        final Environment environment = previousDeployedApplication.getEnvironment();
        GeneratedDeployeds generatedDeployeds = new GeneratedDeployeds();
        Dictionaries dictionaries = of(environment.getDictionaries()).filterBy(newPackage.getApplication());
        DeploymentContext deploymentContext = deployedProcessorsFactory.createContextForUpgrade(appliedDistribution, dictionaries, previousDeployedApplication.getDeployeds());
        DeployedGenerator deployedGenerator = deployedProcessorsFactory.createUpgradeDeployedGenerator();
        Set<Container> containers = previousDeployedApplication.getDeployeds().stream().map(Deployed::getContainer).collect(Collectors.toSet());
        newPackage.getDeployables().stream().forEach(d -> {
            for (Container container : containers) {
                generatedDeployeds.merge(deployedGenerator.generateDeployed(deploymentContext, d, container));
            }
        });

        return generatedDeployeds;
    }

    public DeployedApplication generateUpdateDeployedApplication(DeployedApplication deployedApplication, Version newVersion, Set<String> missingPlaceholdersAggregator) {
        logger.debug("Creating upgrade deployed application from: [{}] with new version [{}]", deployedApplication, newVersion);
        Dictionaries dictionaries = of(deployedApplication.getEnvironment().getDictionaries()).filterBy(newVersion.getApplication());
        return deployedApplicationFactory.createUpgradeDeployedApplication(newVersion, deployedApplication, dictionaries.consolidate(), missingPlaceholdersAggregator);

    }

    public List<Deployed> createDeployedsForNonExistingCombinations(AppliedDistribution appliedDistribution, DistributionVersion version, Domain environment, final Set<Deployed> deployeds, Set<String> missingPlaceholdersAggregator, Map<String, String> userProvidedPlaceholder) {
        Set<? extends Deployable> deployables = version.getDeployables();
        Set<? extends Container> members = environment.getMembers();

        final List<Deployed> validExistingDeployeds = deployeds.stream().filter(input -> input.getDeployable() != null).collect(toList());
        logger.debug("Valid existing deployeds: {}", validExistingDeployeds);

        final List<Deployed> invalidExistingDeployeds = deployeds.stream().filter(input -> input.getDeployable() == null).collect(toList());
        logger.debug("Invalid existing deployeds: {}", invalidExistingDeployeds);

        FilterSet filterSet = new FilterSet(validExistingDeployeds);

        Dictionaries dictionaries = of(environment.getDictionaries()).filterBy(version.getDistribution()).withAdditionalEntries(userProvidedPlaceholder);
        GeneratedDeployeds generatedDeployeds = generateDeployedsOfType(appliedDistribution, deployables, members, null, dictionaries);
        logger.debug("Going to filter previously present deployeds");
        generatedDeployeds.getDeployeds().forEach(filterSet::add);
        missingPlaceholdersAggregator.addAll(generatedDeployeds.getUnresolvedPlaceholders());

        List<Deployed> allValidAsList = filterSet.getAllValidAsList();
        allValidAsList.addAll(invalidExistingDeployeds);
        return allValidAsList;
    }

    private GeneratedDeployeds generateDeployedsOfType(AppliedDistribution appliedDistribution, Collection<? extends Deployable> deployables, Collection<? extends Container> containers, Type deployedType, Dictionaries dictionaries) {
        return generateDeployedsOfType(appliedDistribution, deployables, containers, deployedType, dictionaries, false);
    }

    private GeneratedDeployeds generateDeployedsOfType
            (AppliedDistribution appliedDistribution, Collection<? extends Deployable> deployables, Collection<? extends Container> containers, Type deployedType, Dictionaries dictionaries, boolean ignoreTags) {
        GeneratedDeployeds generatedDeployeds = new GeneratedDeployeds();
        DeployedGenerator deployedGenerator = ignoreTags
                ? deployedProcessorsFactory.createCoreDeployedGenerator()
                : deployedProcessorsFactory.createTagDeployedGenerator();
        DeploymentContext deploymentContext = deployedType != null
                ? deployedProcessorsFactory.createContextForSpecificType(appliedDistribution, dictionaries, deployedType)
                : deployedProcessorsFactory.createContextWithCalculatedType(appliedDistribution, dictionaries);
        for (Deployable deployable : deployables) {
            for (Container container : containers) {
                generatedDeployeds.merge(deployedGenerator.generateDeployed(deploymentContext, deployable, container));
            }
        }
        return generatedDeployeds;
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployedService.class);

    public void scanMissingPlaceholders(DeployedApplication deployedApplication, Environment environment, Set<String> missingPlaceholdersAggregator) {
        deployedApplicationFactory.scanMissingPlaceholders(deployedApplication, environment, missingPlaceholdersAggregator);
    }

    private static class FilterSet {

        private Function<Deployed, Key> KEY_FUNC = input -> new Key(input.getDeployable().getId(), input.getContainer().getId());

        private Map<Key, Value> map = newHashMap();

        /**
         * Create a set with initial data which is all invalid.
         *
         * @param initialDeployeds
         */
        public FilterSet(Collection<Deployed> initialDeployeds) {
            for (Deployed initialDeployed : initialDeployeds) {
                map.put(KEY_FUNC.apply(initialDeployed), new Value(initialDeployed));
            }
        }

        /**
         * Add a new element to the set, if the element previously existed, mark the previous occurrence valid, else add it as a valid element.
         *
         * @param ci
         */
        public void add(ConfigurationItem ci) {
            Deployed<?, ?> deployed = (Deployed<?, ?>) ci;
            Key k = KEY_FUNC.apply(deployed);
            Value v = new Value(deployed).valid();
            Value previous = map.put(k, v);
            // Reset the previous value and validate it.
            if (previous != null) {
                logger.debug("Filtering out [{}] because it is already present.", deployed);
                map.put(k, previous.valid());
            }
        }

        public List<Deployed> getAllValidAsList() {
            return map.values().stream().filter(Value::isValid).map(value -> value.deployed).collect(toList());
        }
    }

    private static class Value {
        private Deployed deployed;
        private boolean valid;

        private Value(Deployed deployed) {
            this.deployed = deployed;
        }

        Value valid() {
            this.valid = true;
            return this;
        }

        private boolean isValid() {
            return valid;
        }

    }

    private static class Key {
        private String deployableId;
        private String containerId;

        private Key(String deployableId, String containerId) {
            this.deployableId = deployableId;
            this.containerId = containerId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            final Key key = (Key) o;

            return containerId.equals(key.containerId) && deployableId.equals(key.deployableId);

        }

        @Override
        public int hashCode() {
            int result = deployableId.hashCode();
            result = 31 * result + containerId.hashCode();
            return result;
        }
    }


}
