package com.xebialabs.deployit.core.upgrade;

import com.xebialabs.deployit.server.api.repository.RawRepository;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;
import com.xebialabs.deployit.upgrade.RawRepositoryImpl;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.*;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.atomic.AtomicInteger;

import static com.google.common.collect.Lists.newArrayList;

public class Deployit401ArchivedTaskDateUtcConversion extends Upgrade {

    private static String pattern = "dd MMM yyyy HH:mm:ss z";

    private static List<String> dateProperties = newArrayList("startDate", "completionDate", "lastModificationDate", "scheduledDate");

    @Override
    public boolean doUpgrade(RawRepository repository) throws UpgradeException {
        logger.info("Starting conversion of Task dates to UTC.");
        Session session = ((RawRepositoryImpl) repository).getSession();
        Node tasks = repository.read("/tasks");
        try {
            NodeIterator nodes = tasks.getNodes();
            convertNodes(nodes, tasks.getName(), session, new AtomicInteger(0));
            logger.info("Conversion of Task dates to UTC complete.");
            return true;
        } catch (RepositoryException e) {
            logger.error("Failed to convert repository",e);
        }
        return false;
    }


    private void convertNodes(NodeIterator nodeIterator, String parent, Session session, AtomicInteger checkpointCounter) throws RepositoryException {
        while (nodeIterator.hasNext()) {
            Node node = nodeIterator.nextNode();
            String nodeId = parent + "/" + node.getName();
            convertNode(node, nodeId, session, checkpointCounter);
            convertNodes(node.getNodes(), nodeId, session, checkpointCounter);
        }
    }

    private void convertNode(Node node, String nodeId, Session session, AtomicInteger checkpointCounter) throws RepositoryException {
        logger.debug("Converting date properties on node [{}]", nodeId);
        boolean converted = false;
        for (String dateProperty : dateProperties) {
            if (node.hasProperty(dateProperty)) {
                logger.debug("Converting date property [{}:{}]", nodeId, dateProperty);
                Property dateProp = node.getProperty(dateProperty);
                Calendar oldDate = dateProp.getDate();
                if (oldDate == null) {
                    logger.warn("Date property [{}:{}] is null and cannot be converted.", nodeId, dateProperty);
                    continue;
                }

                if (!oldDate.getTimeZone().getID().equals("GMT")) {
                    DateTime utcDateTime = new DateTime(oldDate).toDateTime(DateTimeZone.UTC);
                    Calendar newDate = utcDateTime.toGregorianCalendar();
                    logger.info("Converted date property [{}] from [{}] to [{}] on node [{}].", dateProp.getName(), format(oldDate), format(newDate), nodeId);
                    node.setProperty(dateProperty, newDate);
                    converted = true;
                }

            }
        }

        if (converted) {
            if (checkpointCounter.incrementAndGet() >= 100) {
                logger.info("Taking checkpoint...");
                session.save();
                checkpointCounter.set(0);
            }
        }
    }

    private String format(Calendar date) {
        final SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern, Locale.ENGLISH);
        simpleDateFormat.setTimeZone(date.getTimeZone());
        return simpleDateFormat.format(date.getTime());
    }

    @Override
    public Version upgradeVersion() {
        return Version.valueOf("deployit", "4.0.1");
    }

    private static final Logger logger = LoggerFactory.getLogger(Deployit401ArchivedTaskDateUtcConversion.class);

}
