package com.xebialabs.deployit.service.importer;

import static com.google.common.collect.Lists.newArrayList;
import static java.util.Collections.sort;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import com.xebialabs.deployit.plugin.api.utils.TFileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.server.api.importer.ImportedPackage;
import com.xebialabs.deployit.server.api.importer.ImportingContext;
import com.xebialabs.deployit.server.api.importer.ListableImporter;
import com.xebialabs.deployit.server.api.importer.PackageInfo;
import com.xebialabs.deployit.service.importer.reader.DeployableConfigurationItemReader;
import com.xebialabs.deployit.service.importer.reader.ManifestReader;
import com.xebialabs.deployit.util.GuavaFiles;

import de.schlichtherle.truezip.file.TFile;

public class ManifestBasedDarImporter implements ListableImporter {
	public static final String TEMPORARY_FILES = "temporaryFiles";

	@Override
	public List<String> list(File directory) {
		List<String> packages = newArrayList(PackageScanner.scan(directory));
		sort(packages);
		return packages;
	}

	@Override
	public boolean canHandle(ImportSource source) {
		File file = source.getFile();
		return PackageScanner.isDarPackage(file) || PackageScanner.isExplodedPackage(file);
	}

	@Override
	public PackageInfo preparePackage(ImportSource source, ImportingContext ctx) {
		PackageInfo packageInfo = new PackageInfo(source);
		updatePackageInfoWithManifestValues(packageInfo);
		ctx.setAttribute(TEMPORARY_FILES, Lists.<TFile>newArrayList());
		return packageInfo;
	}

	private void updatePackageInfoWithManifestValues(final PackageInfo packageInfo) {
		Manifest manifest = getManifest(packageInfo);
		ManifestReader.verifyPackageFormatVersion(manifest);

		packageInfo.setApplicationName(ManifestReader.getApplicationName(manifest));
		packageInfo.setApplicationVersion(ManifestReader.getApplicationVersion(manifest));
	}

	@Override
	public ImportedPackage importEntities(PackageInfo packageInfo, ImportingContext ctx) {
		ImportedPackage importedPackage = new ImportedPackage(packageInfo);
		Manifest manifest = getManifest(packageInfo);
		createDeployableEntities(packageInfo, manifest.getEntries(), importedPackage, ctx);
		return importedPackage;
	}

	private void createDeployableEntities(PackageInfo packageInfo, Map<String, Attributes> entries, ImportedPackage importedPackage, ImportingContext ctx) {
		DeployableConfigurationItemReader reader = new DeployableConfigurationItemReader(entries);
		for (Map.Entry<String, Attributes> entry : entries.entrySet()) {
			if (reader.isMiddlewareResource(entry)) {
				importedPackage.addDeployable(reader.readMiddlewareConfiguration(importedPackage.getDeploymentPackage(), entry));
			} else {
				importedPackage.addDeployable(reader.readArtifact(importedPackage.getDeploymentPackage(), packageInfo.getSource(), entry, ctx));
			}
		}
	}

	private Manifest getManifest(PackageInfo packageInfo) {
		TFile manifestFileForUnpackedPackage = PackageScanner.getManifestFileFromImportSource(packageInfo.getSource());
		try {
			return ManifestReader.readManifest(manifestFileForUnpackedPackage);
		} finally {
			TFileUtils.umountQuietly(new TFile(packageInfo.getSource().getFile()));
		}
	}

	@Override
	public void cleanUp(PackageInfo packageInfo, ImportingContext ctx) {
		List<TFile> files = ctx.getAttribute(TEMPORARY_FILES);
		for (TFile file : files) {
			logger.debug("Cleaning up temporary file {}", file);
			try {
				File javaFile = file.getFile();
				if (javaFile.isDirectory()) {
					GuavaFiles.deleteRecursively(javaFile);
				} else {
					if (!javaFile.delete()) {
						logger.info("Couldn't delete file: {}", javaFile);
					}
				}				
			} catch (IOException e) {
				logger.error("Couldn't clean up file {}", file, e);
			}
		}
		TFileUtils.umountQuietly(new TFile(packageInfo.getSource().getFile()));
	}

	private static final Logger logger = LoggerFactory.getLogger(ManifestBasedDarImporter.class);
}
