package com.xebialabs.deployit.service.deployment;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;

@Component
public class TypeCalculator {

	public Type findMostSpecificDeployedTypeForDeployableAndContainerTypes(final Type deployableType, final Type containerType) {
		List<Type> deployedTypesForDeployableAndContainer = findDeployedTypesForDeployableAndContainerTypes(deployableType, containerType);

		logger.debug("Found types {} for <{}, {}>", new Object[] {deployedTypesForDeployableAndContainer, deployableType, containerType});

		if (deployedTypesForDeployableAndContainer.isEmpty()) {
			return null;
		}
		if (deployedTypesForDeployableAndContainer.size() == 1) {
			return deployedTypesForDeployableAndContainer.get(0);
		}

		List<Type> copy = filterMostSpecific(deployedTypesForDeployableAndContainer);

		logger.debug("Result of filtering most specific: {}", copy);

		if (copy.size() == 1) {
			return copy.get(0);
		}

		throw new IllegalArgumentException(format("Found %s for %s and %s, expected exactly 1. Before filtering we had: %s", copy, deployableType, containerType, deployedTypesForDeployableAndContainer));
	}

	List<Type> filterMostSpecific(final List<Type> deployedTypesForDeployableAndContainer) {
		List<Type> copy = newArrayList(deployedTypesForDeployableAndContainer);

		for (Type currentType : deployedTypesForDeployableAndContainer) {
			Descriptor currentDescriptor = DescriptorRegistry.getDescriptor(currentType);

			for (Iterator<Type> copyIt = copy.iterator(); copyIt.hasNext();) {
				Type comparedType = copyIt.next();
				Descriptor comparedDescriptor = DescriptorRegistry.getDescriptor(comparedType);

				boolean deployableEquals = comparedDescriptor.getDeployableType().equals(currentDescriptor.getDeployableType());
				boolean containerEquals = comparedDescriptor.getContainerType().equals(currentDescriptor.getContainerType());
				boolean containerSupertype = comparedDescriptor.getContainerType().isSuperTypeOf(currentDescriptor.getContainerType());
				boolean deployableSupertype = comparedDescriptor.getDeployableType().isSuperTypeOf(currentDescriptor.getDeployableType());

				boolean isSupertype = comparedType.isSuperTypeOf(currentType);

				String currentTypeString = format("%s<%s, %s>", currentType, currentDescriptor.getDeployableType(), currentDescriptor.getContainerType());
				String comparedTypeString = format("%s<%s, %s>", comparedType, comparedDescriptor.getDeployableType(), comparedDescriptor.getContainerType());
				if (containerSupertype && deployableEquals) {
					logger.debug("Filtering out [{}] because is has a container supertype wrt. [{}]", comparedTypeString, currentTypeString);
					copyIt.remove();
				} else if (deployableSupertype && containerEquals) {
					logger.debug("Filtering out [{}] because is has a deployable supertype wrt. [{}]", comparedTypeString, currentTypeString);
					copyIt.remove();
				} else if (deployableSupertype && containerSupertype) {
					logger.debug("Filtering out [{}] because is has a container and deployable supertype wrt. [{}]", comparedTypeString, currentTypeString);
					copyIt.remove();
				} else if (isSupertype) {
					logger.debug("Filtering out [{}] because is a supertype of [{}]", comparedTypeString, currentTypeString);
					copyIt.remove();
				}
			}
		}
		return copy;
	}

	public List<Type> findDeployedTypesForDeployableAndContainerTypes(final Type deployableType, final Type containerType) {
		Collection<Type> deployedTypes = DescriptorRegistry.getSubtypes(Type.valueOf(Deployed.class));
		final Descriptor deployableDesc = DescriptorRegistry.getDescriptor(deployableType);
		final Descriptor containerDesc = DescriptorRegistry.getDescriptor(containerType);
		return newArrayList(Collections2.filter(deployedTypes, new Predicate<Type>() {
			@Override
			public boolean apply(Type input) {
				Descriptor deployedDesc = DescriptorRegistry.getDescriptor(input);
				return !deployedDesc.isVirtual() && deployableDesc.isAssignableTo(deployedDesc.getDeployableType()) && containerDesc.isAssignableTo(deployedDesc.getContainerType());
			}
		}));
	}

	private static final Logger logger = LoggerFactory.getLogger(TypeCalculator.class);
}
