package com.xebialabs.deployit.security.authentication;

import com.google.common.collect.MapMaker;
import com.xebialabs.deployit.conversion.Base64Coder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;

import java.nio.charset.Charset;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import static com.xebialabs.deployit.security.SecurityUtils.getAuthentication;

public class RememberMe {
    
    private static final ConcurrentMap<String, Authentication> AUTH_CACHE = new MapMaker().expireAfterAccess(10, TimeUnit.MINUTES).makeMap();

    public static void remember(Authentication auth) {
        String key = createKey(auth);
        logger.debug("Remembering authentication token for [{}]", auth.getName());
        AUTH_CACHE.put(key, auth);
    }

    public static void forget(Authentication auth) {
        String key = createKey(auth);
        if (AUTH_CACHE.containsKey(key)) {
            AUTH_CACHE.remove(key);
        } else {
            logger.info("Couldn't forget authentication for [{}] because it was not remembered.", auth.getName());
        }
    }

    public static Authentication getRemembered(String username, String password) {
        String key = encodeUserPassword(username, password);
        if (AUTH_CACHE.containsKey(key)) {
            return AUTH_CACHE.get(key);
        } else {
            logger.debug("Token for [{}] not yet remembered.", username);
            return null;
        }
    }

    public static void forgetMe() {
        forget(getAuthentication());
    }

    private static String createKey(Authentication auth) {
        if (auth instanceof UsernamePasswordAuthenticationToken) {
            UsernamePasswordAuthenticationToken token = (UsernamePasswordAuthenticationToken) auth;
            String user = token.getName();
	        String password = token.getCredentials().toString();
            return encodeUserPassword(user, password);
        } else {
            throw new IllegalArgumentException("The authentication should always be a UsernamePasswordAuthenticationToken");
        }
    }

    private static String encodeUserPassword(String user, String password) {
        return new String(new Base64Coder().encode((user + ":" + password).getBytes()), Charset.defaultCharset());
    }

    private static final Logger logger = LoggerFactory.getLogger(RememberMe.class);
}
