package com.xebialabs.deployit.repository;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.io.Closeables.closeQuietly;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.DATA_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.FILENAME_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_DATE_PROPERTY_NAME;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;

import com.xebialabs.deployit.io.Imploder;
import com.xebialabs.deployit.repository.core.Securable;
import com.xebialabs.deployit.security.permission.Permission;
import org.apache.jackrabbit.value.ReferenceValue;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.util.PasswordObfuscator;
import com.xebialabs.overthere.OverthereFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Writes an {@link ConfigurationItem} to a JCR {@link Node}.
 */
class NodeWriter {

	private final Session session;
	private final ConfigurationItem item;
	private final Node node;
	private boolean basicsWritten = false;

	NodeWriter(final Session session, final ConfigurationItem item, final Node node) {
		this.session = session;
		this.item = item;
		this.node = node;
	}

	void writeBasics() throws RepositoryException {
		copyConfigurationItemTypeName();
		basicsWritten = true;
	}

	void write() throws RepositoryException {
		if (!basicsWritten) {
			writeBasics();
		}
		copyData();
		copyValuesIntoNode();
		copyMetadata();
	}

	private void copyConfigurationItemTypeName() throws RepositoryException {
		node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, item.getType().toString());
	}

	private void copyData() throws RepositoryException {
		if (item instanceof SourceArtifact) {
			OverthereFile file = ((Artifact) item).getFile();
			if (file != null) {
				node.setProperty(FILENAME_PROPERTY_NAME, file.getName());
				InputStream dataInput;
				if (item instanceof FolderArtifact && file.isDirectory()) {
					try {
						byte[] implode = Imploder.implode(file);
						dataInput = new ByteArrayInputStream(implode);
					} catch (IOException e) {
						throw new RuntimeIOException("Could not zip up the data in file " + file);
					}
				} else {
					dataInput = file.getInputStream();
				}
				try {
					Binary binary = session.getValueFactory().createBinary(dataInput);
					node.setProperty(DATA_PROPERTY_NAME, binary);
				} finally {
					closeQuietly(dataInput);
				}
			}
		}
	}

	private void copyValuesIntoNode() throws RepositoryException {
		Descriptor ciDescriptor = DescriptorRegistry.getDescriptor(item.getType());
		Collection<PropertyDescriptor> propertyDescriptors = Collections2.filter(ciDescriptor.getPropertyDescriptors(), new Predicate<PropertyDescriptor>() {
			@Override
			public boolean apply(PropertyDescriptor input) {
				return !input.isTransient();
			}
		});
		for (PropertyDescriptor pd : propertyDescriptors) {
			if (pd.get(item) == null) {
				removePropertyFromNode(pd);
				continue;
			}

			logger.trace("Writing property [{}] with value [{}]", pd.getFqn(), pd.isPassword() ? "********" : pd.get(item));

			switch (pd.getKind()) {
			case BOOLEAN:
			case INTEGER:
			case STRING:
			case ENUM:
				copyPrimitivePropertyIntoNode(pd);
				break;
			case SET_OF_STRING:
				copyCollectionOfStringsPropertyIntoNode(pd);
				break;
			case SET_OF_CI:
				copyCollectionOfConfigurationItemsPropertyIntoNode(pd);
				break;
			case LIST_OF_STRING:
				copyCollectionOfStringsPropertyIntoNode(pd);
				break;
			case LIST_OF_CI:
				copyCollectionOfConfigurationItemsPropertyIntoNode(pd);
				break;
			case CI:
				copyConfigurationItemPropertyIntoNode(pd);
				break;
			case MAP_STRING_STRING:
				copyMapPropertyIntoNode(pd);
				break;
			default:
				throw new IllegalArgumentException("Cannot convert property " + pd.getName() + " because it is of unsupported kind " + pd.getKind());

			}
		}
	}

	@SuppressWarnings("unchecked")
    private void copyMapPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
		Map<String, String> value = (Map<String, String>) pd.get(item);
		List<Value> valueList = newArrayList();
		for (String s : value.keySet()) {
			Value v = session.getValueFactory().createValue(s + "=" + value.get(s));
			valueList.add(v);
		}

		Value[] jcrValues = valueList.toArray(new Value[valueList.size()]);
		node.setProperty(pd.getName(), jcrValues);
	}

	private void copyMetadata() throws RepositoryException {
		node.setProperty(LAST_MODIFIED_DATE_PROPERTY_NAME, Calendar.getInstance());
	}

	private void removePropertyFromNode(PropertyDescriptor pd) throws RepositoryException {
		try {
			node.getProperty(pd.getName()).remove();
		} catch (PathNotFoundException ignored) {
		}
	}

	private void copyPrimitivePropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
		String valueAsString = pd.get(item).toString();
		if(pd.getKind() == STRING && pd.isPassword()) {
			valueAsString = PasswordObfuscator.ensureEncrypted(valueAsString);
		}
		node.setProperty(pd.getName(), valueAsString);
	}

	private void copyCollectionOfStringsPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
		final ValueFactory valueFactory = session.getValueFactory();

		Collection<?> valueAsCollection = convertObjectValueToCollection(pd, pd.get(item));
		List<Value> jcrValueList = newArrayList();
		for (Object each : valueAsCollection) {
			if (!(each instanceof String)) {
				throw new IllegalArgumentException("Element in property " + pd.getName() + " of repository entity " + item.getId()
				        + " is not a String: " + each);
			}
			String stringValue = (String) each;
			jcrValueList.add(valueFactory.createValue(stringValue));
		}
		Value[] jcrValues = jcrValueList.toArray(new Value[jcrValueList.size()]);
		node.setProperty(pd.getName(), jcrValues);
	}

	private void copyConfigurationItemPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
		if (pd.isAsContainment()) {
			return;
		}

		String referencedCiId = ((ConfigurationItem) pd.get(item)).getId();
		Node referencedCi = session.getNode(getAbsolutePathFromId(referencedCiId));
		node.setProperty(pd.getName(), referencedCi);
	}

	private void copyCollectionOfConfigurationItemsPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
		if (pd.isAsContainment()) {
			return;
		}

		Collection<?> valueAsCollection = convertObjectValueToCollection(pd, pd.get(item));
		List<Value> jcrReferenceList = newArrayList();
		for (Object each : valueAsCollection) {
			if (!(each instanceof ConfigurationItem)) {
				throw new IllegalArgumentException("Element in property " + pd.getName() + " of repository entity " + item.getId()
				        + " is not a ConfigurationItem: " + each);
			}
			String referencedCiId = ((ConfigurationItem) each).getId();
			Node referencedCi = session.getNode(getAbsolutePathFromId(referencedCiId));
			jcrReferenceList.add(new ReferenceValue(referencedCi));
		}
		Value[] jcrReferenceValues = jcrReferenceList.toArray(new Value[jcrReferenceList.size()]);
		node.setProperty(pd.getName(), jcrReferenceValues);
	}

	private Collection<?> convertObjectValueToCollection(PropertyDescriptor pd, Object value) {
		if (!(value instanceof Collection)) {
			throw new IllegalArgumentException("Property " + pd.getName() + " of repository entity " + item.getId() + " is not a Collection: " + value);
		}
		return (Collection<?>) value;
	}
	
	private static final Logger logger = LoggerFactory.getLogger(NodeWriter.class);
}
