package com.xebialabs.deployit.core.upgrade;

import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrUtils;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.repository.JcrPathHelper;
import com.xebialabs.deployit.repository.core.Directory;
import com.xebialabs.deployit.repository.core.Securable;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.server.api.repository.RawRepository;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.Lists.newArrayList;

public class Deployit371GlobalToLocalPermissions extends Upgrade {
	private static final List<String> TO_BE_CONVERTED = newArrayList(Permission.TASK_SKIPSTEP.getPermissionName(), Permission.TASK_MOVESTEP.getPermissionName());

	@Override
	public boolean doUpgrade(RawRepository repository) throws UpgradeException {
		logger.info("*** Starting Deployit 3.7.1 upgrade -- Global to Local permissions ***");
		Node securityNode = repository.read(JcrConstants.SECURITY_NODE_ID);
		try {
			Map<String, String> permissionToRoleIds = readGlobalPermissions(securityNode);
			logger.debug("Found global permissions: {}", permissionToRoleIds);
			convertPermissions(repository, permissionToRoleIds);
			writeNewGlobalPermissions(securityNode, permissionToRoleIds);
		} catch (RepositoryException e) {
			throw new UpgradeException("Could not convert global to local permissions", e);
		}
		logger.info("*** Done Deployit 3.7.1 upgrade -- Global to Local permissions ***");
		return true;
	}

	private void convertPermissions(RawRepository repository, Map<String, String> permissionToRoleIds) throws RepositoryException {
		Map<String, String> toBeConverted = Maps.filterKeys(permissionToRoleIds, new Predicate<String>() {
			public boolean apply(String input) {
				return TO_BE_CONVERTED.contains(input);
			}
		});
		logger.debug("Going to write these permissions under the local nodes: {}", toBeConverted);
		Node environments = repository.read(JcrPathHelper.getAbsolutePathFromId(Metadata.ConfigurationItemRoot.ENVIRONMENTS.getRootNodeName()));
		updateNode(environments, toBeConverted);
		List<Node> directories = repository.findNodesByType(Type.valueOf(Directory.class));
		logger.debug("Found these directories {}", directories);
		for (Node directory : directories) {
			updateNode(directory, toBeConverted);
		}
	}

	private void updateNode(Node node, Map<String, String> toBeConverted) throws RepositoryException {
		if (node.hasProperty(Securable.SECURITY_PERMISSIONS_PROPERTY) &&
				Permission.TASK_MOVESTEP.isApplicableTo(JcrPathHelper.getIdFromAbsolutePath(node.getPath()))) {
			logger.debug("Writing new local permissions on node [{}]", node.getPath());
			Map<String, String> map = JcrUtils.readMap(node.getProperty(Securable.SECURITY_PERMISSIONS_PROPERTY));
			logger.debug("read permissions {}", map);
			for (String k : toBeConverted.keySet()) {
				if (map.containsKey(k)) {
					map.put(k, Joiner.on(",").join(map.get(k), toBeConverted.get(k)));
				} else {
					map.put(k, toBeConverted.get(k));
				}
			}
			logger.debug("Going to write permissions {}", map);
			JcrUtils.writeMap(node, Securable.SECURITY_PERMISSIONS_PROPERTY, map);
		}
	}

	private void writeNewGlobalPermissions(Node securityNode, Map<String, String> permissionToRoleIds) throws RepositoryException {
		Map<String, String> keepGlobal = Maps.filterKeys(permissionToRoleIds, new Predicate<String>() {
			public boolean apply(String input) {
				return !TO_BE_CONVERTED.contains(input);
			}
		});
		JcrUtils.writeMap(securityNode, Securable.SECURITY_PERMISSIONS_PROPERTY, keepGlobal);
	}

	private Map<String, String> readGlobalPermissions(Node securityNode) throws RepositoryException {
		Property property = securityNode.getProperty(Securable.SECURITY_PERMISSIONS_PROPERTY);
		return JcrUtils.readMap(property);
	}

	@Override
	public Version upgradeVersion() {
		return Version.valueOf("deployit", "3.7.1");
	}

	private static final Logger logger = LoggerFactory.getLogger(Deployit371GlobalToLocalPermissions.class);
}
