package com.xebialabs.deployit.task;

import com.xebialabs.deployit.event.Event;
import com.xebialabs.deployit.event.EventBus;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.task.TaskStepInfo.StepState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.security.core.Authentication;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.FutureTask;

import static com.google.common.base.Preconditions.checkState;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.task.Task.State.*;

@SuppressWarnings("serial")
public abstract class Task implements Serializable, Runnable {

	private static final String MDC_KEY_TASK_ID = "taskId";

	private String id;

	private String label = "Task #" + getClass();

	private String owner;

	private transient Authentication ownerCredentials;

	private Calendar startDate;

	private Calendar completionDate;

	private final StepList steps;

    private volatile boolean stopRequested;

	private volatile transient FutureTask<Object> wrappingTask;

	private volatile State state = PENDING;

	public enum State {
		PENDING, QUEUED, EXECUTING, DONE, STOPPED, CANCELLED
	}

    public Task(final List<? extends Step<?>> steps) {
        this(new StepList(steps), PENDING);
	}

	Task(final StepList stepList, final State state) {
		this.steps = stepList;
		this.state = state;
	}

	protected boolean executeSteps() {
		ExecutionContextAttributes context = new ExecutionContextAttributes();
		try {
			while (steps.hasMoreSteps()) {
				TaskStep currentStep = steps.getNextStep();
				currentStep.execute(context);
				if (currentStep.isFailed()) {
					steps.rollback();
					return false;
				}

				// Step is DONE, but thread has been interrupted or has been requested to stop, do not continue now!
				if (isAborted() || isStopped()) {
					return false;
				}
			}
		} finally {
			context.destroy();
		}

		return true;
	}

	public void moveStep(int stepNr, int newPosition) {
		checkState(state == State.PENDING, "Cannot move steps when task is not pending anymore.");
		steps.move(stepNr, newPosition);
	}

	@Override
	public void run() {
		MDC.put(MDC_KEY_TASK_ID, getId());
		try {
			stopRequested = false;
			checkState(!isExecuting(), "task is already executing");
			checkState(wrappingTask != null, "task should have a wrapper set!");
	
			if (!isReadyForExecution())
				return;
	
			if (state == QUEUED) {
				startDate = Calendar.getInstance();
			}
	
			try {
				setState(EXECUTING);
				final boolean result = executeSteps();
				logger.debug("Executed all steps of {}", this);
				if (!result && isAborted()) {
					setState(STOPPED);
				} else if (!result && steps.hasMoreSteps()) {
					setState(STOPPED);
				} else {
					setState(DONE);
				}
			} finally {
				// I'm done, don't need my wrappingTask anymore!
				wrappingTask = null;
				// Clear the abort flag
				Thread.interrupted();
			}
		} finally {
			MDC.remove(MDC_KEY_TASK_ID);
		}
	}

    public int getFailureCount() {
        int failureCount = 0;
        for(TaskStep step : steps) {
            failureCount += step.getFailureCount();
        }
        return failureCount;
    }

	/**
	 * override this method if something needs to be executed/saved/deleted after the state is changed to DONE
	 */
	public void doAfterTaskStateChangedToDone() {
	}

	/**
	 * override this method if something needs to be executed/saved/deleted after the state is changed to ABORTED
	 */
	public void doAfterTaskStateChangedToAborted() {
	}

     /**
     * Override this method when pre-flight checks need to be performed for task.
     */
    public void performPreFlightChecks() {

    }

	public void stop() {
		logger.info("Stop requested for {}", this);
		stopRequested = true;
	}

	/**
	 * Aborts a task when it is executing.
	 */
	public void abort() {
		if (isExecuting()) {
			wrappingTask.cancel(true);
		} else {
			logger.info("Cannot abort task [{}] which is not currently executing.", id);
		}
	}

	/**
	 * Cancels a task that has not started executing or that has stopped executing (but is not DONE).
	 */
	public void cancel() {
		checkArgument(isReadyForExecution() || state == PENDING, "Can only cancel a task that is PENDING or STOPPED");
		setState(CANCELLED);
	}

	/**
	 * Destroys a task. Cleans up any temporary files needed for execution of the task.
	 */
	public void destroy() {
	}

	public boolean isReadyForExecution() {
		return state == QUEUED || state == STOPPED;
	}

	public boolean isExecuting() {
		return state == EXECUTING;
	}

	private boolean isStopped() {
		return stopRequested;
	}

	private boolean isAborted() {
		return wrappingTask.isCancelled() || Thread.currentThread().isInterrupted();
	}

	public List<TaskStep> getSteps() {
		return Collections.unmodifiableList(steps);
	}

	public void setWrappingTask(final FutureTask<Object> wrappingTask) {
		this.wrappingTask = wrappingTask;
	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public String getLabel() {
		return label;
	}

	public void setLabel(String label) {
		this.label = label;
	}

	public String getOwner() {
		return owner;
	}

	public void setOwner(final String owner) {
		this.owner = owner;
	}

	public Authentication getOwnerCredentials() {
		return ownerCredentials;
	}

	public void setOwnerCredentials(final Authentication ownerCredentials) {
		this.ownerCredentials = ownerCredentials;
	}

	public Calendar getStartDate() {
		return startDate;
	}

	void setStartDate(final Calendar startDate) {
		this.startDate = startDate;
	}

	public Calendar getCompletionDate() {
		return completionDate;
	}

	void setCompletionDate(final Calendar completionDate) {
		this.completionDate = completionDate;
	}

	public int getCurrentStepNr() {
		return steps.getCurrentStepNr();
	}

	public int getNrOfSteps() {
		return steps.getNrOfSteps();
	}

	public TaskStep getStep(final int stepNr) {
		return steps.getStep(stepNr);
	}

	public FutureTask<Object> getWrappingTask() {
		return wrappingTask;
	}

	public State getState() {
		return state;
	}

	public void processAfterRecovery() {
		for (TaskStep eachStep : getSteps()) {
			if (eachStep.getState() == StepState.EXECUTING) {
				eachStep.setLog(eachStep.getLog() + "Step was executing when the server crashed. Try executing it again.\n");
				eachStep.transitionExecutingStateToFailedState();
				steps.rollback();
				break;
			}
		}
		setState(STOPPED);
	}

	void setState(final State state) {
		if (state == this.state) {
			return;
		}

		boolean shouldDestroy = false;

        this.state = state;

		switch (state) {
        case QUEUED:
            logger.info("{} is now queued.", this);
            break;
		case EXECUTING:
			logger.info("Execution of {} has started", this);
			break;
		case STOPPED:
			logger.error("Execution of {} was stopped", this);
			completionDate = Calendar.getInstance();
			try {
				doAfterTaskStateChangedToAborted();
			} catch (RuntimeException e) {
				logger.error("Execution of {} has stopped because of an internal error", this, e);
				this.state = STOPPED;
				throw e;
			}
			break;
		case CANCELLED:
			logger.info("Execution of {} has been cancelled", this);
			completionDate = Calendar.getInstance();
			shouldDestroy = true;
			break;
		case DONE:
			logger.info("Execution of {} has completed successfully", this);
			completionDate = Calendar.getInstance();
			shouldDestroy = true;
			try {
				doAfterTaskStateChangedToDone();
			} catch (RuntimeException e) {
				logger.error("Execution of {} has stopped because of an internal error", this, e);
				this.state = STOPPED;
				throw e;
			}
			break;
		}

		try {
			EventBus.publish(new TaskStateChangeEvent(this, this.state));
		} catch (RuntimeException e) {
			logger.error("Exception while publishing TaskStateChangeEvent: ", e);
			throw e;
		}

		if (shouldDestroy) {
			destroy();
		}
	}

	@Override
	public String toString() {
		return "task [" + id + "]";
	}
	
    public class TaskStateChangeEvent implements Event {
		private final Task task;
		private final State oldState;

		public TaskStateChangeEvent(Task task, final State oldState) {
			this.task = task;
			this.oldState = oldState;
		}

		public Task getTask() {
			return task;
		}

		public State getOldState() {
			return oldState;
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(Task.class);
}
