package com.xebialabs.deployit.security;

import com.google.common.base.Function;
import com.google.common.base.Predicates;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.repository.JcrPathHelper;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.util.Option;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static com.xebialabs.deployit.security.Permissions.*;
import static com.xebialabs.deployit.util.Option.none;
import static com.xebialabs.deployit.util.Option.some;

@Component
public class PermissionEnforcer {

	public static final String ROLE_ADMIN = "ROLE_ADMIN";

	private JcrTemplate jcrTemplate;
	private final RoleService roleService;

	@Autowired
	public PermissionEnforcer(JcrTemplate jcrTemplate, RoleService roleService) {
		this.jcrTemplate = jcrTemplate;
		this.roleService = roleService;
	}

	public boolean hasPermission(Authentication authentication, final Permission... permissions) {
		return hasPermission(authentication, newArrayList(permissions), "");
	}

	public boolean hasLoggedInUserPermission(final Permission... permissions) {
		Authentication authentication = getAuthentication();
		return hasPermission(authentication, newArrayList(permissions), "");
	}

	public boolean hasLoggedInUserPermission(Permission permission, String onConfigurationItem) {
		return hasLoggedInUserPermission(newArrayList(permission), onConfigurationItem);
	}

	public boolean hasLoggedInUserPermission(final List<Permission> permissions, String onConfigurationItem) {
		return hasPermission(getAuthentication(), permissions, onConfigurationItem);
	}

	private boolean hasPermission(Authentication auth, List<Permission> permissions, String onConfigurationItem) {
		final Collection<String> allPrincipals = authenticationToPrincipals(auth);
		List<Role> allRoles = roleService.getRolesFor(auth);
		if (isAdmin(allPrincipals, allRoles)) return true;

		return checkPermission(permissions, onConfigurationItem, allRoles);
	}

	private boolean checkPermission(final List<Permission> permissions, final String onConfigurationItem, final List<Role> allRoles) {
		return jcrTemplate.execute(new JcrCallback<Boolean>() {
			public Boolean doInJcr(Session session) throws IOException, RepositoryException {
				if (Strings.nullToEmpty(onConfigurationItem).isEmpty()) {
					return checkGlobalPermission(session, permissions, allRoles);
				} else {
					return checkCiPermission(session, permissions, onConfigurationItem, allRoles);
				}

			}
		});
	}

	private boolean checkGlobalPermission(Session session, List<Permission> permissions, List<Role> allRoles) throws RepositoryException {
		Node node = session.getNode(JcrConstants.SECURITY_NODE_ID);

		Option<Boolean> o = checkPermission(node, permissions, allRoles);
		if (o.isSet()) {
			return o.get();
		}
		return false;
	}

	private boolean checkCiPermission(Session session, List<Permission> permissions, String onConfigurationItem, List<Role> allRoles) throws RepositoryException {
		List<String> nodes = nodesToInspect(onConfigurationItem);
		for (String nodeId : nodes) {
			if (!session.nodeExists(nodeId)) continue;
			Option<Boolean> o = checkCiPermission(session.getNode(nodeId), permissions, allRoles);
			if (o.isSet()) {
				return o.get();
			}
		}

		return false;
	}

	private Option<Boolean> checkCiPermission(Node node, List<Permission> permissions, List<Role> allRoles) throws RepositoryException {
		if (!isSecurable(node)) {
			return none();
		}

		return checkPermission(node, permissions, allRoles);
	}

	private Option<Boolean> checkPermission(Node node, List<Permission> allPermissions, List<Role> allRoles) throws RepositoryException {
		logger.debug("Trying to read permissions from [{}]", node.getPath());
		Map<String, String> permissions = readPermissionMap(node);

		if (permissions.isEmpty()) {
			logger.debug("[{}] has no permissions set, checking up the tree.", node.getPath());
			return none();
		}

		logger.debug("[{}] has permissions enabled, which are: [{}]", node.getPath(), permissions);
		for (Permission permission : allPermissions) {
			if (permissions.containsKey(permission.getPermissionName())) {
				Iterable<Integer> permittedRoles = splitRoles(permissions.get(permission.getPermissionName()));
				if (Iterables.any(permittedRoles, Predicates.in(newArrayList(rolesToIds(allRoles))))) {
					return some(true);
				}
			}
		}
		return some(false);
	}

	private List<String> nodesToInspect(String onConfigurationItem) {
		return newArrayList(transform(getFullTreeAsSeparateNodesInChildToRootOrder(onConfigurationItem), new Function<String, String>() {
			public String apply(String input) {
				return JcrPathHelper.getAbsolutePathFromId(input);
			}
		}));
	}

	boolean isAdmin(Collection<String> allPrincipals, List<Role> allRoles) {
		Boolean isAdmin = allPrincipals.contains(ROLE_ADMIN) || checkPermission(newArrayList(Permission.ADMIN), "", allRoles);
		logger.debug("Admin privileges [{}] granted to {}", isAdmin ? "are" : "are not", allPrincipals);
		return isAdmin;
	}

	private static final Logger logger = LoggerFactory.getLogger(PermissionEnforcer.class);
}
