package com.xebialabs.deployit.util;

import static com.xebialabs.deployit.checks.Checks.checkArgument;

import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.SecretKeySpec;

import com.xebialabs.deployit.conversion.Base64Coder;
import com.xebialabs.deployit.exception.DeployitException;

public class PasswordObfuscator {
    private static final byte[] KEY_BYTES = new byte[]{
            (byte) 0xe3, (byte) 0xea, 0x16, (byte) 0xaf, (byte) 0x94, 0x5f, 0x5b, 0x17,
            0x2e, (byte) 0xe6, 0x31, (byte) 0xd3, (byte) 0xc0, 0x70, 0x27, (byte) 0xd6
    };
    private static final SecretKeySpec AES_KEY = new SecretKeySpec(KEY_BYTES, "AES");
    private static final Base64Coder BASE_64_CODER = new Base64Coder();

    public static boolean isEncoded(String password) {
        try {
            BASE_64_CODER.decode(password.getBytes());
            return true;
        } catch (RuntimeException e) {
            return false;
        }
    }

	public static boolean isEncrypted(String password) {
		if (isEncoded(password)) {
			byte[] decode = BASE_64_CODER.decode(password.getBytes());
			try {
				Cipher cipher = getCipher(Cipher.DECRYPT_MODE);
				cipher.doFinal(decode);

				return true;
			} catch (GeneralSecurityException e) {
				throw new IllegalStateException("Could not decrypt Base64 encoded password");
			}
		}
		return false;
	}

    public static String encrypt(String password) {
        checkArgument(password != null, "The password is null");
        checkArgument(!password.startsWith(BASE_64_CODER.identifier()), "The password contains the illegal sequence " + BASE_64_CODER.identifier());

        try {
	        Cipher aes = getCipher(Cipher.ENCRYPT_MODE);
            return new String(BASE_64_CODER.encode(aes.doFinal(password.getBytes())));
        } catch (GeneralSecurityException e) {
            throw new DeployitException("Could not encrypt the password using AES 128");
        }
    }

	public static String ensureEncrypted(String password) {
		if (isEncrypted(password)) {
			return password;
		} else {
			return encrypt(password);
		}
	}

    public static String decrypt(String encryptedPassword) {
        checkArgument(encryptedPassword != null, "The password is null");
        checkArgument(encryptedPassword.startsWith(BASE_64_CODER.identifier()), "The password should be Base64 encoded");

        try {
	        Cipher aes = getCipher(Cipher.DECRYPT_MODE);
            return new String(aes.doFinal(BASE_64_CODER.decode(encryptedPassword.getBytes())));
        } catch (GeneralSecurityException e) {
            throw new DeployitException("Could not decrypt the password using AES 128");
        }
    }

	public static String ensureDecrypted(String password) {
		if (isEncrypted(password)) {
			return decrypt(password);
		} else {
			return password;
		}
	}

	private static Cipher getCipher(int mode) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException {
		Cipher aes = Cipher.getInstance("AES");
		aes.init(mode, AES_KEY);
		return aes;
	}

}
