package com.xebialabs.deployit.service.importer;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.plugin.api.reflect.ReflectionsHolder;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.repository.ConfigurationItemData;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.server.api.importer.*;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.replacement.MustachePlaceholderScanner;
import com.xebialabs.deployit.service.validation.Validator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.io.File;
import java.lang.reflect.Modifier;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Lists.newArrayList;

public class ImporterServiceImpl implements ImporterService {

    private RepositoryService repositoryService;

    private Validator validator;

    private File importablePackageDirectory;

	private List<Importer> importers = Lists.newArrayList();

	@Autowired
	public ImporterServiceImpl(RepositoryService repositoryService, Validator validator) {
		this.repositoryService = repositoryService;
		this.validator = validator;
	}

	@PostConstruct
	public void initImporters() {
		Set<Class<? extends Importer>> importers = ReflectionsHolder.getSubTypesOf(Importer.class);
		logger.debug("Found importers: {}", importers);
		for (Class<? extends Importer> importer : importers) {
			try {
				if (importer.isInterface() || Modifier.isAbstract(importer.getModifiers()))
					continue;
				logger.debug("Importer {} registered.", importer);
				this.importers.add(importer.newInstance());
			} catch (Exception e) {
				throw new IllegalStateException("Could not instantiate importer: " + importer, e);
			}
		}
		Collections.sort(this.importers, new Comparator<Object>() {
			@Override
			public int compare(Object o1, Object o2) {
				return o1.getClass().getSimpleName().compareTo(o2.getClass().getSimpleName());
			}
		});

		logger.info("Importers configured in Deployit: {}", this.importers);
	}

    public void setImportablePackageDirectory(final File importablePackageDirectory) {
        this.importablePackageDirectory = importablePackageDirectory;
    }

	@Override
	public File getImportablePackageDirectory() {
		return importablePackageDirectory;
	}
	
	@Override
	public List<String> listPackages() {
		List<String> packagesFound = Lists.newArrayList();
		for (Importer importer : importers) {
			if (importer instanceof ListableImporter) {
				packagesFound.addAll(((ListableImporter) importer).list(importablePackageDirectory));
			}
		}

		Collections.sort(packagesFound);
		return packagesFound;
	}

	@Override
	public String importPackage(ImportSource source) {
		try {
			for (Importer importer : importers) {
				if (importer.canHandle(source)) {
					return doImport(source, importer);
				}
			}
			throw new ImporterException("Unable to obtain the package to import from the import source");
		} finally {
			source.cleanUp();
		}
	}

	private String doImport(ImportSource source, Importer importer) {
		ImportingContext ctx = new DefaultImportingContext();
		PackageInfo packageInfo = importer.preparePackage(source, ctx);
		boolean isUpgrade = isUpgrade(packageInfo);
		checkPermission(isUpgrade, packageInfo);
		checkImported(packageInfo);
		try {
			ImportedPackage importedPackage = importer.importEntities(packageInfo, ctx);
			scanPlaceholders(importedPackage);
			List<ConfigurationItem> toCreate = createEntities(importedPackage, isUpgrade);
			validate(toCreate);
			repositoryService.create(toCreate.toArray(new ConfigurationItem[toCreate.size()]));
			return importedPackage.getVersion().getId();
		} finally {
			importer.cleanUp(packageInfo, ctx);
		}
	}

	private void scanPlaceholders(ImportedPackage importedPackage) {
		if (importedPackage.getVersion() instanceof DeploymentPackage) {
			for (Deployable deployable : importedPackage.getDeployables()) {
				if (deployable instanceof SourceArtifact) {
					((SourceArtifact) deployable).initPlaceholders(new MustachePlaceholderScanner());
				}
			}
		}
	}

	private void checkImported(PackageInfo packageInfo) {
		if (repositoryService.exists(IdGenerator.generateId(packageInfo.getApplicationId(), packageInfo.getApplicationVersion()))) {
			throw new ImporterException("Already imported version %s of application %s", packageInfo.getApplicationVersion(), packageInfo.getApplicationName());
		}
	}

	private List<ConfigurationItem> createEntities(ImportedPackage importedPackage, boolean isUpgrade) {
		List<ConfigurationItem> toCreate = newArrayList();
		if (!isUpgrade) {
			toCreate.add(importedPackage.getApplication());
		}
		toCreate.add(importedPackage.getVersion());
		if (importedPackage.getVersion() instanceof DeploymentPackage) {
			toCreate.addAll(importedPackage.getDeployables());
		}
		return toCreate;
	}

	private void validate(List<ConfigurationItem> toCreate) {
		List<ValidationMessage> msgs = newArrayList();
		for (ConfigurationItem toCreateEntity : toCreate) {
			msgs.addAll(validator.validate(toCreateEntity, toCreate));
		}

		if (!msgs.isEmpty()) {
			throw new ImporterException("Import failed with the following validation errors %s", msgs.toString());
		}
	}

	private boolean isUpgrade(PackageInfo packageInfo) {
		List<ConfigurationItemData> applications = repositoryService.list(new SearchParameters().setType(Type.valueOf(Application.class)));
		String app = null;
		for (ConfigurationItemData application : applications) {
			if (subStringAfterLast(application.getId(), "/").equals(packageInfo.getApplicationName())) {
				checkState(app == null, "Found more than 1 udm.Application with the same name: [%s] and [%s]", app, application);
				app = application.getId();
			}
		}
		if (app != null) {
			packageInfo.setApplicationName(app.substring(Metadata.ConfigurationItemRoot.APPLICATIONS.getRootNodeName().length() + 1));
			return true;
		}

		return false;
	}

	private String subStringAfterLast(String s, String sep) {
		if (s.contains(sep)) {
			return s.substring(s.lastIndexOf(sep) + sep.length());
		}
		return s;
	}
	
	private void checkPermission(boolean isUpgrade, PackageInfo packageInfo) {
		if (isUpgrade) {
			checkPermission(Permission.IMPORT_UPGRADE, packageInfo.getApplicationId());
		} else {
			checkPermission(Permission.IMPORT_INITIAL, packageInfo.getApplicationId());
		}
	}

	void checkPermission(Permission permission, String onConfigurationItems) {
		if (!permission.getPermissionHandler().hasPermission(onConfigurationItems)) {
			throw PermissionDeniedException.forPermission(permission, onConfigurationItems);
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(ImporterServiceImpl.class);
}
