package com.xebialabs.deployit.repository;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_DATE_PROPERTY_NAME;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;
import static java.lang.String.format;
import static javax.jcr.query.Query.JCR_SQL2;

import java.util.Calendar;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;

import com.xebialabs.deployit.checks.Checks;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;

/**
 */
public class SearchQueryBuilder {
	public static final String CI_SELECTOR_NAME = "ci";

	private static final String EQUALITY_OPERATOR = "=";
	private static final String LIKE_OPERATOR = "LIKE";
	private static final String NAME_IS_OPERATOR = "NAME_IS";
	private static final String NAME_LIKE_OPERATOR = "NAME_LIKE";
	private static final String ISCHILDNODE_OPERATOR = "ISCHILDNODE";

	private final SearchParameters parameters;
	private final StringBuilder joins;
	private final List<Condition> conditions;
	private int nextSelectorId = 1;

	public SearchQueryBuilder(final SearchParameters parameters) {
		this.parameters = parameters;
		this.joins = new StringBuilder();
		this.conditions = newArrayList();

		Checks.checkArgument(parameters.name == null || !parameters.name.contains("\'"), "Name [%s] contains a single quote (\')", parameters.name);
		Checks.checkArgument(parameters.parent == null || !parameters.parent.contains("\'"), "Parent [%s] contains a single quote (\')", parameters.parent);
	}

	public Query build(final Session session) throws RepositoryException {
		createJoinsAndConditions();

		QueryManager qm = session.getWorkspace().getQueryManager();
		final ValueFactory valueFactory = session.getValueFactory();
		final String q = constructQueryString();
		final Query query = qm.createQuery(q, JCR_SQL2);

		addBinds(valueFactory, query);
		addPagingInfo(query);

		logger.debug("JCR query built: {}", query.getStatement());
		return query;
	}

	private String constructQueryString() {
		StringBuilder builder = new StringBuilder();
		builder.append("SELECT " + CI_SELECTOR_NAME + ".* FROM [" + CONFIGURATION_ITEM_NODETYPE_NAME + "] AS " + CI_SELECTOR_NAME);
		builder.append(joins.toString());

		if (!conditions.isEmpty()) {
			builder.append(" WHERE ");
			builder.append(Joiner.on(" AND ").join(Collections2.transform(conditions, new Function<Condition, String>() {
				@Override
				public String apply(final Condition input) {
					return input.build();
				}
			})));
		}

		builder.append(" ORDER BY NAME(" + CI_SELECTOR_NAME +")");

		logger.debug("Query string built: {}", builder.toString());

		return builder.toString();
	}

	private void addBinds(final ValueFactory valueFactory, final Query query) throws RepositoryException {
		for (Condition eachCondition : conditions) {
			if (eachCondition.parameter == null)
				continue;

			for (int i = 0; i < eachCondition.values.length; i++) {
				Object eachValue = eachCondition.values[i];
				Value v;
				if (eachValue instanceof String) {
					v = valueFactory.createValue((String) eachValue);
				} else if (eachValue instanceof Calendar) {
					v = valueFactory.createValue((Calendar) eachValue);
				} else {
					throw new IllegalArgumentException("Value of Condition is not a String or a Calendar but a " + eachValue.getClass().getName());
				}
				query.bindValue(eachCondition.parameter + i, v);

				logger.debug("Bound {} to {}", eachCondition.parameter + i, v.getString());
			}
		}
	}

	void addPagingInfo(final Query query) {
		if (parameters.resultsPerPage > 0) {
			query.setLimit(parameters.resultsPerPage);
			query.setOffset(parameters.page * parameters.resultsPerPage);
		}
	}

	private void createJoinsAndConditions() {
		createConditionForParent();
		createConditionForName();
		createConditionForConfigurationItemTypeName();
		createConditionForDate();
		createConditionsForProperties();
	}

	private void createConditionForConfigurationItemTypeName() {
		if (parameters.type != null) {
			List<String> types = Lists.newArrayList();
			types.add(parameters.type.toString());
			for (Type subtype : DescriptorRegistry.getSubtypes(parameters.type)) {
				types.add(subtype.toString());
			}
			conditions.add(Condition.from(CI_SELECTOR_NAME, CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, "_configurationItemTypeName", EQUALITY_OPERATOR, false,
			        (Object[]) types.toArray(new String[types.size()])));
		}
	}

	private void createConditionForParent() {
		if (!Strings.nullToEmpty(parameters.parent).trim().isEmpty()) {
			String parent = Strings.nullToEmpty(parameters.parent);
			if (!parent.startsWith("/")) {
				parent = "/" + parent;
			}
			conditions.add(Condition.isChildNode(CI_SELECTOR_NAME, parent));
		}
	}

	private void createConditionForName() {
		if (!Strings.nullToEmpty(parameters.name).trim().isEmpty()) {
			if (parameters.name.contains("%")) {
				conditions.add(Condition.nameLike(CI_SELECTOR_NAME, parameters.name));
			} else {
				conditions.add(Condition.nameIs(CI_SELECTOR_NAME, parameters.name));
			}
		}
	}

	private void createConditionForDate() {
		if (parameters.before != null) {
			conditions.add(Condition.from(CI_SELECTOR_NAME, LAST_MODIFIED_DATE_PROPERTY_NAME, "_before", "<=", parameters.before));
		}
	}

	private void createConditionsForProperties() {
		for (Map.Entry<String, String> entry : parameters.properties.entrySet()) {
			createConditionForProperty(entry.getKey(), entry.getValue());
		}
	}

	private void createConditionForProperty(final String propertyName, final String propertyValue) {
		if (parameters.type != null) {
			Descriptor descriptor = DescriptorRegistry.getDescriptor(parameters.type);
			if (descriptor != null) {
				PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(propertyName);
				if (propertyDescriptor != null) {
					if (propertyDescriptor.getKind() == CI) {
						String selectorName = "referenced" + nextSelectorId;
						joins.append(" INNER JOIN [" + CONFIGURATION_ITEM_NODETYPE_NAME + "] AS " + selectorName + " ON " + CI_SELECTOR_NAME + ".["
						        + propertyName + "] = " + selectorName + ".[jcr:uuid]");
						int lastSlash = propertyValue.lastIndexOf('/');
						checkArgument(lastSlash != -1, propertyValue + " is a ID but does not contain a slash (/)");
						final String parent = getAbsolutePathFromId(propertyValue.substring(0, lastSlash));
						final String name = propertyValue.substring(lastSlash + 1);
						conditions.add(Condition.isChildNode(selectorName, parent));
						conditions.add(Condition.nameIs(selectorName, name));
						nextSelectorId++;
						return;
					} else if (propertyDescriptor.getKind() == SET_OF_CI || propertyDescriptor.getKind() == LIST_OF_CI) {
						// FIXME: Find out how to make the JOIN works on this
						// case. The INNER JOIN used for the single CI property
						// causes unpredictable results.
						// The results of the JOIN are different every time if
						// the multi-value property has more than one value.
						throw new IllegalArgumentException("Cannot query property " + propertyDescriptor + " because it is of type SET_OF_CIS");
					}
				}
			}
		}
		conditions.add(Condition.match(CI_SELECTOR_NAME, propertyName, propertyName, propertyValue));
	}

	private static class Condition {
		String selector;
		String field;
		String operator;
		String parameter;
		Object[] values;
		boolean caseInsensitive;

		private static final String NAME_IS_FORMAT = "NAME(%s) = \'%s\'";
		private static final String NAME_LIKE_FORMAT = "LOWER(NAME(%s)) LIKE \'%s\'";
		private static final String ISCHILDNODE_FORMAT = "ISCHILDNODE(%s, [\'%s\'])";
		private static final String CASE_SENSITIVE = "%s.[%s] %s $%s";
		private static final String CASE_INSENSITIVE = "LOWER(%s.[%s]) %s $%s";

		String build() {
			StringBuilder conditionString = new StringBuilder();
			conditionString.append("(");
			for (int i = 0; i < values.length; i++) {
				if (i > 0) {
					conditionString.append(" OR ");
				}
				
				if (operator.equals(NAME_IS_OPERATOR)) {
					conditionString.append(format(NAME_IS_FORMAT, selector, values[i]));
				} else if (operator.equals(NAME_LIKE_OPERATOR)) {
						conditionString.append(format(NAME_LIKE_FORMAT, selector, values[i]));
				} else if (operator.equals(ISCHILDNODE_OPERATOR)) {
					conditionString.append(format(ISCHILDNODE_FORMAT, selector, values[i]));
				} else {
					conditionString.append(format(caseInsensitive ? CASE_INSENSITIVE : CASE_SENSITIVE, selector, field, operator, parameter + i));
				}
			}
			conditionString.append(")");
			return conditionString.toString();
		}

		static Condition match(final String selector, final String field, final String parameter, final String value) {
			String operator;
			boolean caseInsensitive;
			if (value.contains("%")) {
				operator = LIKE_OPERATOR;
				caseInsensitive = true;
				
			} else {
				operator = EQUALITY_OPERATOR;
				caseInsensitive = false;
			}
			return from(selector, field, parameter, operator, caseInsensitive, value);
		}

		static Condition nameIs(final String selector, final String value) {
			return from(selector, null, null, NAME_IS_OPERATOR, false, value);
		}

		static Condition nameLike(final String selector, final String value) {
			return from(selector, null, null, NAME_LIKE_OPERATOR, false, value.toLowerCase());
		}

		static Condition isChildNode(final String selector, final String value) {
			return from(selector, null, null, ISCHILDNODE_OPERATOR, false, value);
		}

		static Condition from(final String selector, final String field, final String parameter, final String operator, final Object value) {
			return from(selector, field, parameter, operator, false, value);
		}

		static Condition from(final String selector, final String field, final String parameter, final String operator, final boolean caseInsensitive,
		        final Object... values) {
			Condition t = new Condition();
			t.selector = selector;
			t.field = field;
			t.parameter = parameter;
			t.operator = operator;
			t.caseInsensitive = caseInsensitive;
			t.values = values;
			return t;
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(SearchQueryBuilder.class);
}
