package com.xebialabs.deployit.task.jcrarchive;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.Task;

import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import javax.jcr.query.QueryManager;
import javax.jcr.query.qom.*;
import java.util.Calendar;
import java.util.List;

import static com.xebialabs.deployit.task.jcrarchive.JcrTaskArchive.*;
import static javax.jcr.query.qom.QueryObjectModelConstants.*;

public class JcrArchivedTaskSearchQueryBuilder extends ArchivedTaskSearchParameters {

    private ValueFactory vf;
    private QueryObjectModelFactory qf;

    static final String TASK_SELECTOR_NAME = "task";

    public JcrArchivedTaskSearchQueryBuilder(QueryManager qm, ValueFactory vf, ArchivedTaskSearchParameters p) {
        super(p);
        this.vf = vf;
        this.qf = qm.getQOMFactory();
    }

    public QueryObjectModel buildQuery() throws RepositoryException {
        List<Constraint> andConstraints = Lists.newArrayList();

        appendTaskUuidCriteria(andConstraints);
        appendEnvironmentApplicationAndVersionCriteria(andConstraints);
        appendExecutedByCriteria(andConstraints);
        appendStatusCriteria(andConstraints);
        appendDateRangeCriteria(andConstraints);

        Selector selector = qf.selector(JcrConstants.TASK_NODETYPE_NAME , TASK_SELECTOR_NAME);
        return qf.createQuery(selector, createSingleAndConstraint(andConstraints), null, null);
    }

    private void appendTaskUuidCriteria(List<Constraint> constraints) throws RepositoryException {
        if (isNotBlank(taskUuid))
            constraints.add(equalConstraint(JcrConstants.ID_PROPERTY_NAME,taskUuid));
    }

	private boolean isNotBlank(String s) {
		return !Strings.nullToEmpty(s).trim().isEmpty();
	}

	private void appendEnvironmentApplicationAndVersionCriteria(List<Constraint> constraints) throws RepositoryException {
        StringBuilder descendentOf = new StringBuilder(JcrConstants.TASKS_NODE_ID);
        boolean applicationInDescendentPath = false;
        boolean versionInDescendentPath = false;

        if (isNotBlank(environment)) {
            descendentOf.append("/").append(environment);
            if (isNotBlank(application)) {
                applicationInDescendentPath = true;
                descendentOf.append("/").append(application);
                if (isNotBlank(version)) {
                    versionInDescendentPath = true;
                    descendentOf.append("/").append(version);
                }
            }

	        constraints.add(qf.descendantNode(TASK_SELECTOR_NAME, descendentOf.toString()));
        }

        if (!applicationInDescendentPath && isNotBlank(application)) {
            constraints.add(equalConstraint(APPLICATION, application));
        }

        if (!versionInDescendentPath && isNotBlank(version)) {
            constraints.add(equalConstraint(VERSION, version));
        }
    }

    private void appendExecutedByCriteria(List<Constraint> constraints) throws RepositoryException {
        if (isNotBlank(executedBy)) {
            constraints.add(equalConstraint(OWNING_USER, executedBy));
        }
    }

    private void appendStatusCriteria(List<Constraint> constraints) throws RepositoryException {
        switch (status) {
            case COMPLETED:
	            constraints.add(qf.and(equalConstraint(STATE, Task.State.DONE.name()), equalConstraint(FAILURE_COUNT, 0l)));
                break;
            case CANCELLED:
                constraints.add(equalConstraint(STATE, Task.State.CANCELLED.name()));
                break;
            case COMPLETED_AFTER_RETRY:
                Constraint stateConstraint = equalConstraint(STATE, Task.State.DONE.name());
                Constraint failureConstraint = greaterThanConstraint(FAILURE_COUNT, 0l);
                constraints.add(qf.and(stateConstraint, failureConstraint));
                break;
        }
    }

	private void appendDateRangeCriteria(List<Constraint> constraints) throws RepositoryException {
        switch (dateRangeSearch) {
            case AFTER:
                Calendar startDateAsCal = cloneAndSetTimeToStartOfDay(startDate);
                constraints.add(greaterThanOrEqualConstraint(START_DATE, startDateAsCal));
                break;

            case BEFORE:
                Calendar endDateAsCal = cloneAndSetTimeToEndOfDay(endDate);
                constraints.add(lessThanOrEqualConstraint(START_DATE, endDateAsCal));
                break;

            case BETWEEN:
                Calendar betweenStartDate = cloneAndSetTimeToStartOfDay(startDate);
                Calendar betweenEndDate = cloneAndSetTimeToEndOfDay(endDate);
                Constraint startConstraint = greaterThanOrEqualConstraint(START_DATE, betweenStartDate);
                Constraint endConstraint = lessThanOrEqualConstraint(START_DATE, betweenEndDate);
                constraints.add(startConstraint);
                constraints.add(endConstraint);
        }
    }

    private Constraint equalConstraint(String propName, String value) throws RepositoryException {
	    return addConstraint(propName, JCR_OPERATOR_EQUAL_TO, vf.createValue(value));
    }

	private Constraint equalConstraint(String propName, long value) throws RepositoryException {
		return addConstraint(propName, JCR_OPERATOR_EQUAL_TO, vf.createValue(value));
	}

	private Constraint greaterThanConstraint(String propName, long value) throws RepositoryException {
		return addConstraint(propName, JCR_OPERATOR_GREATER_THAN, vf.createValue(value));
	}

	private Constraint greaterThanOrEqualConstraint(String propName, Calendar value) throws RepositoryException {
		return addConstraint(propName, JCR_OPERATOR_GREATER_THAN_OR_EQUAL_TO, vf.createValue(value));
	}

    private Constraint lessThanOrEqualConstraint(String propName, Calendar value) throws RepositoryException {
	    return addConstraint(propName, JCR_OPERATOR_LESS_THAN_OR_EQUAL_TO, vf.createValue(value));
    }

	private Constraint addConstraint(String propName, String op, Value v) throws RepositoryException {
		return qf.comparison(qf.propertyValue(TASK_SELECTOR_NAME, propName), op, qf.literal(v));
	}

    private Constraint createSingleAndConstraint(List<Constraint> constraints) throws RepositoryException {
	    if (constraints.isEmpty()) return null;
        Constraint andConstraint = constraints.get(0);
        for (int i = 1; i < constraints.size(); i++) {
            andConstraint = qf.and(andConstraint, constraints.get(i));
        }
        return andConstraint;
    }

    private Calendar cloneAndSetTimeToStartOfDay(Calendar date) {
        Calendar startOfDay = (Calendar) date.clone();
        startOfDay.set(Calendar.HOUR_OF_DAY, 0);
        startOfDay.set(Calendar.MINUTE, 0);
        startOfDay.set(Calendar.SECOND, 0);
        startOfDay.set(Calendar.MILLISECOND, 0);
        return startOfDay;
    }

    private Calendar cloneAndSetTimeToEndOfDay(Calendar date) {
        Calendar endOfDay = (Calendar) date.clone();
        endOfDay.set(Calendar.HOUR_OF_DAY, 23);
        endOfDay.set(Calendar.MINUTE, 59);
        endOfDay.set(Calendar.SECOND, 59);
        endOfDay.set(Calendar.MILLISECOND, 999);
        return endOfDay;
    }
}
