package com.xebialabs.deployit.task.deployment;

import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.repository.ChangeSet;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.security.SecurityCallback;
import com.xebialabs.deployit.security.SecurityTemplate;
import com.xebialabs.deployit.security.UsernameAndPasswordCredentials;
import com.xebialabs.deployit.task.DeploymentTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

import static com.google.common.collect.Lists.newArrayList;

@SuppressWarnings("serial")
public class UpgradeDeploymentTask extends DeploymentTask {

    private DeltaSpecification specification;
    private DeployedApplication oldApp;

    public UpgradeDeploymentTask(final DeltaSpecification specification, final DeployedApplication oldApp, List<? extends DeploymentStep> steps, RepositoryService repositoryService, WorkDir oldWorkDir, WorkDir newWorkDir) {
		super(steps, repositoryService, oldWorkDir, newWorkDir);
        this.specification = specification;
        this.oldApp = oldApp;
        initFromDeployment(specification.getDeployedApplication());
    }

    @Override
    public void performPreFlightChecks() {
        UsernameAndPasswordCredentials owner = getOwnerCredentials();
		SecurityTemplate.executeAs(owner.getUsername(), owner.getPassword(), new SecurityCallback<Object>() {
            @Override
            public Object doAs() {
                repositoryService.checkReferentialIntegrity(determineRepositoryChanges());
                return null;
            }
        });
    }

    private ChangeSet determineRepositoryChanges() {
        final DeployedApplication newDeployment = specification.getDeployedApplication();

        // Save the mappings that are stored as subnodes of the deployment that was just created
        final List<ConfigurationItem> createRoes = newArrayList();
        final List<ConfigurationItem> updateRoes = newArrayList();
        final List<String> deleteRoeIds = newArrayList();

        for (Delta operation : specification.getDeltas()) {
            switch (operation.getOperation()) {
                case CREATE:
                    createRoes.add(operation.getDeployed());
                    newDeployment.addDeployed(operation.getDeployed());
                    logger.info("The CI {} was {} in the repository", operation.getDeployed().getId(), operation.getOperation());
                    break;
                case DESTROY:
                    oldApp.getDeployeds().remove(operation.getPrevious());
                    updateRoes.add(oldApp);
                    deleteRoeIds.add(operation.getPrevious().getId());
                    logger.info("The CI {} was {} in the repository", operation.getPrevious().getId(), operation.getOperation());
                    break;
                case MODIFY:
	            case NOOP:
                    updateRoes.add(operation.getDeployed());
                    newDeployment.addDeployed(operation.getDeployed());
                    logger.info("The CI {} was {} in the repository", operation.getDeployed().getId(), operation.getOperation());
                    break;
            }
        }

        // Update the deployment to include the creating task id and to refer to the mappings that were just created
//        RepositoryObjectEntity updatedNewDeploymentEntity = context.toEntity(newDeployment);
//        updatedNewDeploymentEntity.setCreatingTaskId(getId());
        updateRoes.add(newDeployment);

        ChangeSet changeset = new ChangeSet();
        changeset.setDeleteCiIds(deleteRoeIds);
        changeset.setCreateCis(createRoes);
        changeset.setUpdateCis(updateRoes);

        return changeset;
    }

	@Override
	public void doAfterTaskStateChangedToDone() {
		UsernameAndPasswordCredentials owner = getOwnerCredentials();
		SecurityTemplate.executeAs(owner.getUsername(), owner.getPassword(), new SecurityCallback<Object>() {
			@Override
			public Object doAs() {
                repositoryService.execute(determineRepositoryChanges());
				return null;
			}
		});
	}

    private static final Logger logger = LoggerFactory.getLogger(UpgradeDeploymentTask.class);

    @Override
    public DeploymentType getDeploymentType() {
        return DeploymentTask.DeploymentType.UPGRADE;
    }
}
