package com.xebialabs.deployit.repository;

import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.io.Closeables.closeQuietly;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.DATA_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.FILENAME_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ID_PROPERTY_NAME;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;
import java.util.Set;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.query.Query;
import javax.jcr.query.QueryResult;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.ByteStreams;
import com.google.common.io.InputSupplier;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.plugin.DerivedArtifactFile;
import com.xebialabs.deployit.plugin.Exploder;
import com.xebialabs.deployit.plugin.FileWithoutContent;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.util.PasswordObfuscator;
import com.xebialabs.overthere.local.LocalFile;

class NodeReader {
	private Session session;
	private Node node;
	private final WorkDir workDir;
	private NodeReaderContext context;

	private NodeReader(Session session, Node node, WorkDir workDir, NodeReaderContext context) {
		this.session = session;
		this.node = node;
		this.workDir = workDir;
		this.context = context;
	}
	
	static <T extends ConfigurationItem> T read(Session session, Node node, WorkDir workDir) throws RepositoryException {
		NodeReaderContext c = NodeReaderContext.get().hold();
		try {
			return new NodeReader(session, node, workDir, c).<T>read();
		} finally {
			c.release();
		}
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
    private <T extends ConfigurationItem> T read() throws RepositoryException {
		String id = node.getProperty(ID_PROPERTY_NAME).getString();
		logger.trace("{} loading with workdir {}", id, workDir);
		if (context.hasItem(node.getIdentifier())) {
			T item = (T) context.get(node.getIdentifier());
			Descriptor descriptor = DescriptorRegistry.getDescriptor(item.getType());
			if (descriptor.isAssignableTo(SourceArtifact.class) && workDir != null && ((SourceArtifact) item).getFile() instanceof FileWithoutContent) {
				copyData(item, descriptor);
			} else if (descriptor.isAssignableTo(DerivedArtifact.class) && workDir != null && ((DerivedArtifact) item).getFile() == null) {
				copyData(item, descriptor);
			}

			return item;
		}

		final Type type = Type.valueOf(node.getProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString());
		Descriptor descriptor = DescriptorRegistry.getDescriptor(type);

		T item = (T) descriptor.newInstance();
		item.setId(id);
		context.put(node.getIdentifier(), item);

		// TODO add when needed
//		entity.setLastModified(node.getProperty(LAST_MODIFIED_DATE_PROPERTY_NAME).getDate());
//		if (node.hasProperty(CREATING_TASK_ID_PROPERTY_NAME)) {
//			entity.setCreatingTaskId(node.getProperty(CREATING_TASK_ID_PROPERTY_NAME).getString());
//		}
		copyValues(item, descriptor);
		copyData(item, descriptor);
		return item;
	}

	@SuppressWarnings("rawtypes")
    private <T extends ConfigurationItem> void copyData(T item, Descriptor descriptor) throws RepositoryException {
		if (descriptor.isAssignableTo(SourceArtifact.class) && workDir != null) {
			String filename = node.getProperty(FILENAME_PROPERTY_NAME).getString();
			LocalFile localFile = workDir.newFile(filename);
			final Binary binary = node.getProperty(DATA_PROPERTY_NAME).getBinary();
			try {
				logger.debug("Reading Source Artifact {} of type {} data. " + item.getId(), descriptor);
				if (descriptor.isAssignableTo(FolderArtifact.class)) {
					localFile.mkdir();
					try {
						Exploder.explode(createInputSupplier(binary), localFile.getFile());
					} catch (IOException e) {
						throw new RuntimeIOException("Could not extract data from " + item.getId() + " to directory " + localFile.getPath(), e);
					}
				} else {
					OutputStream outputStream = localFile.getOutputStream();
					try {
						ByteStreams.copy(createInputSupplier(binary), outputStream);
					} catch (IOException e) {
						throw new RuntimeIOException("Could not copy data from " + item.getId() + " to file " + localFile.getPath(), e);
					} finally {
						closeQuietly(outputStream);
					}
				}
			} finally {
				binary.dispose();
			}
			((Artifact) item).setFile(localFile);
		} else if (descriptor.isAssignableTo(SourceArtifact.class)) {
			String filename = node.getProperty(FILENAME_PROPERTY_NAME).getString();
			((Artifact) item).setFile(new FileWithoutContent(filename));
		} else if (descriptor.isAssignableTo(DerivedArtifact.class) && workDir != null) {
			DerivedArtifact d = (DerivedArtifact) item;
			if (d.getSourceArtifact() != null) {
				d.setFile(DerivedArtifactFile.create(d));
			}
		}
	}

	private InputSupplier<InputStream> createInputSupplier(final Binary binary) {
		return new InputSupplier<InputStream>() {
			@Override
			public InputStream getInput() throws IOException {
				try {
					return binary.getStream();
				} catch (RepositoryException e) {
					throw new IOException("Could not get inputstream from node.");
				}
			}
		};
	}

	private <T extends ConfigurationItem> void copyValues(T item, Descriptor descriptor) throws RepositoryException {
		for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
			// Skip over default values
			if (!node.hasProperty(pd.getName()) && !pd.isAsContainment()) {
				continue;
			}

			switch (pd.getKind()) {
				case BOOLEAN:
				case INTEGER:
				case STRING:
				case ENUM:
					setPrimitiveProperty(item, pd);
					break;
				case SET_OF_STRING:
					setSetOfStringsProperty(item, pd);
					break;
				case SET_OF_CI:
					setSetOfConfigurationItemsProperty(item, pd);
					break;
				case CI:
					setConfigurationItemProperty(item, pd);
					break;
				case MAP_STRING_STRING:
					copyMapPropertyFromNode(item, pd);
					break;
			}
		}
	}

	private <T extends ConfigurationItem> void copyMapPropertyFromNode(T item, PropertyDescriptor pd) throws RepositoryException {
		Map<String, String> map = newHashMap();
		for (Value value : node.getProperty(pd.getName()).getValues()) {
			String v = value.getString();
			int i = v.indexOf("=");
			map.put(v.substring(0, i), v.substring(i + 1));
		}
		pd.set(item, map);
	}

	private <T extends ConfigurationItem> void setConfigurationItemProperty(T item, PropertyDescriptor pd) throws RepositoryException {
		Node referencedNode;
		if (pd.isAsContainment()) {
			referencedNode = node.getParent();
		} else {
			Value value = node.getProperty(pd.getName()).getValue();
			referencedNode = NodeUtils.getReferencedCiNode(node, value, session);
		}
		pd.set(item, NodeReader.read(session, referencedNode, workDir));
	}

	private <T extends ConfigurationItem> void setSetOfConfigurationItemsProperty(T item, PropertyDescriptor pd) throws RepositoryException {
		Set<ConfigurationItem> items = newHashSet();
		if (pd.isAsContainment()) {
			SearchParameters params = new SearchParameters().setParent(item.getId()).setType(pd.getReferencedType());
			SearchQueryBuilder builder = new SearchQueryBuilder(params);
			final Query query = builder.build(session);
			final QueryResult queryResult = query.execute();
			NodeIterator iterator = queryResult.getNodes();
			while (iterator.hasNext()) {
				Node referencedNode = iterator.nextNode();
				items.add(NodeReader.read(session, referencedNode, workDir));
			}
		} else {
			for (Value each : node.getProperty(pd.getName()).getValues()) {
				Node referencedNode = NodeUtils.getReferencedCiNode(node, each, session);
				items.add(NodeReader.read(session, referencedNode, workDir));
			}
		}
		pd.set(item, items);
	}

	private <T extends ConfigurationItem> void setSetOfStringsProperty(T item, PropertyDescriptor pd) throws RepositoryException {
		Set<String> set = newHashSet();
		for (Value v : node.getProperty(pd.getName()).getValues()) {
			set.add(v.getString());
		}
		pd.set(item, set);
	}

	private <T extends ConfigurationItem> void setPrimitiveProperty(T item, PropertyDescriptor pd) throws RepositoryException {
		String valueAsString = node.getProperty(pd.getName()).getString();
		if(pd.getKind() == STRING && pd.isPassword()) {
			valueAsString = PasswordObfuscator.ensureDecrypted(valueAsString);
		}
		pd.set(item, valueAsString);
	}

	private static final Logger logger = LoggerFactory.getLogger(NodeReader.class);
}
