package com.xebialabs.deployit.repository;

import com.google.common.base.Function;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import javax.jcr.*;
import javax.jcr.nodetype.NodeType;
import javax.jcr.query.Query;
import javax.jcr.query.QueryResult;
import javax.jcr.query.RowIterator;
import javax.jcr.version.VersionManager;
import java.io.IOException;
import java.util.*;

import static com.google.common.base.Joiner.on;
import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.jcr.JcrConstants.*;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.NESTED;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;

@Component
public class JcrRepositoryService implements RepositoryService {

    private JcrTemplate jcrTemplate;

    @Autowired
    public JcrRepositoryService(JcrTemplate jcrTemplate) {
        this.jcrTemplate = jcrTemplate;
    }

    @Override
    public void execute(ChangeSet changeset) {
        jcrTemplate.execute(new ChangeSetExecutor(changeset));
    }

    @Override
    public void checkReferentialIntegrity(final ChangeSet changeset) throws ItemInUseException, ItemAlreadyExistsException {
        jcrTemplate.execute(new ReferentialIntegrityChecker(changeset));
    }

    @Override
    public <T extends ConfigurationItem> void create(final T... entities) {
        ChangeSet changeset = new ChangeSet();
        changeset.setCreateCis(Lists.<ConfigurationItem>newArrayList(entities));
        execute(changeset);
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id) {
        checkNotNull(id, "id is null");

        return this.<T>_read(id, null);
    }

    @Override
    public <T extends ConfigurationItem> T read(String id, WorkDir workDir) {
        checkNotNull(id, "id is null");

        return this.<T>_read(id, workDir);
    }

    private <T extends ConfigurationItem> T _read(final String id, final WorkDir workDir) {
        checkNotNull(id, "id is null");

        return jcrTemplate.execute(new JcrCallback<T>() {
            @Override
            public T doInJcr(Session session) throws IOException, RepositoryException {
                try {
                    Node node = session.getNode(getAbsolutePathFromId(id));
                    return NodeReader.<T>read(session, node, workDir);
                } catch (PathNotFoundException exc) {
                    throw new NotFoundException(exc, "Repository entity [%s] not found", id);
                }
            }
        });
    }

    @Override
    public <T extends ConfigurationItem> void update(final T... cis) {
        ChangeSet changeset = new ChangeSet();
        changeset.setUpdateCis(Lists.<ConfigurationItem>newArrayList(cis));
        execute(changeset);
    }

    @Override
    public void delete(final String... ids) {
        ChangeSet changeset = new ChangeSet();
        changeset.setDeleteCiIds(Lists.<String>newArrayList(ids));
        execute(changeset);
    }

    @Override
    public List<String> list(final SearchParameters parameters) {
        checkNotNull(parameters);

        List<ConfigurationItem> entities = listEntities(parameters);
	    return Lists.transform(entities, new Function<ConfigurationItem, String>() {
		    public String apply(ConfigurationItem input) {
			    return input.getId();
		    }
	    });
    }

    @Override
    public <T extends ConfigurationItem> List<T> listEntities(final SearchParameters parameters) {
        checkNotNull(parameters, "parameters is null");
        return jcrTemplate.execute(new JcrCallback<List<T>>() {
            @Override
            public List<T> doInJcr(final Session session) throws IOException, RepositoryException {
                final Query query = new SearchQueryBuilder(parameters).build(session);
                final QueryResult queryResult = query.execute();
                if(parameters.at != null) {
                	return getDistinctEntities(parameters, session, queryResult);
                } else {
                	return getOrderedEntities(parameters, session, queryResult);
                }
            }
        });
    }

    private <T extends ConfigurationItem> List<T> getDistinctEntities(final SearchParameters parameters, final Session session, final QueryResult queryResult) throws RepositoryException, IOException {
	    Map<String, Pair<T, Node>> nodes = new LinkedHashMap<String, Pair<T, Node>>();
	    final RowIterator iterator = queryResult.getRows();
	    while (iterator.hasNext()) {
	        // TODO temporary fix!
	        // We need to fix the security model for deployment to
	        // inherit the privileges from both the source and the
	        // target.
	        // Remove the try-catch once that is implemented.
	        try {
	            Node each = iterator.nextRow().getNode(SearchQueryBuilder.CI_SELECTOR_NAME);
		        Pair<T, Node> pair = new Pair<T, Node>(NodeReader.<T>read(session, each, null), each);
		        if (nodeNotFoundYetOrNewerThanNodeAlreadyFound(nodes, pair)) {
			        nodes.put(pair.left.getId(), pair);
		        }
	        } catch (RepositoryException rre) {
	            // Ignore: We weren't allowed to read a node or one of it's relations.
	        }
	    }
	    return newArrayList(Collections2.transform(nodes.values(), new Function<Pair<T, Node>, T>() {
		    public T apply(Pair<T, Node> input) {
			    return input.left;
		    }
	    }));
    }

    private <T extends ConfigurationItem> boolean nodeNotFoundYetOrNewerThanNodeAlreadyFound(final Map<String, Pair<T, Node>> entities, final Pair<T, Node> pair) throws RepositoryException {
	    String path = pair.left.getId();
        return !entities.containsKey(path)
                || getLastModified(entities.get(path).right).before(getLastModified(pair.right));
    }

	private Calendar getLastModified(Node node) throws RepositoryException {
		return node.getProperty(LAST_MODIFIED_DATE_PROPERTY_NAME).getDate();
	}

	private <T extends ConfigurationItem> List<T> getOrderedEntities(final SearchParameters parameters, final Session session, final QueryResult queryResult) throws RepositoryException, IOException {
		List<Pair<T, Node>> nodes = newArrayList();
	    final RowIterator iterator = queryResult.getRows();
	    while (iterator.hasNext()) {
	        // TODO temporary fix!
	        // We need to fix the security model for deployment to
	        // inherit the privileges from both the source and the
	        // target.
	        // Remove the try-catch once that is implemented.
	        try {
	            Node each = iterator.nextRow().getNode(SearchQueryBuilder.CI_SELECTOR_NAME);
		        nodes.add(new Pair<T, Node>(NodeReader.<T>read(session, each, null), each));
	        } catch (RepositoryException rre) {
	            // Ignore: We weren't allowed to read a node or one of it's relations.
	        }
	    }

		// Because JCR always returns the current node before any versioned node (while we want it last) and because reverse sorting by the last-modified-date
		// property does not seem to work, we have to implement sorting by last-modified-date in Java. This does not increase memory usage, but will be slower
		// when a lot of entities are returned. Luckily this only happens when specifying an after or a before condition because sorting by ID _is_ handled by
		// JCR.
	    if(parameters.after != null || parameters.before != null) {
		    Collections.sort(nodes, new Comparator<Pair<T, Node>>() {
			    @Override
			    public int compare(Pair<T, Node> lhs, Pair<T, Node> rhs) {
				    int diff = 0;
				    try {
					    diff = lhs.left.getId().compareTo(rhs.left.getId());
						if(diff != 0) {
							return diff;
						} else {
							return getLastModified(lhs.right).compareTo(getLastModified(rhs.right));
						}
				    } catch (RepositoryException e) {
					    throw new IllegalStateException("Could not get path from node.", e);
				    }
			    }
		    });
	    }

	    return newArrayList(Lists.transform(nodes, new Function<Pair<T, Node>, T>() {
		    public T apply(Pair<T, Node> input) {
			    return input.left;
		    }
	    }));
    }

    @Override
    public boolean checkNodeExists(final String id) {
        return jcrTemplate.execute(new JcrCallback<Boolean>() {
            @Override
            public Boolean doInJcr(final Session session) throws IOException, RepositoryException {
                return session.itemExists(getAbsolutePathFromId(id));
            }
        });
    }

	static String generateJcrPropertyNameForNestedProperty(PropertyDescriptor pd, PropertyDescriptor nestedPd) {
        return pd.getName() + "_" + nestedPd.getName();
    }

    private static class ChangeSetExecutor implements JcrCallback<Object> {

        private final ChangeSet changeset;

        ChangeSetExecutor(ChangeSet changeset) {
            checkNotNull(changeset);
            this.changeset = changeset;
        }

        @Override
        public Object doInJcr(Session session) throws IOException, RepositoryException {
            execute(session, true);
            return null;
        }

		public void execute(Session session, boolean autocommit) throws RepositoryException {
			try {
				createEntities(changeset.getCreateCis(), session);
				updateEntities(changeset.getUpdateCis(), session);
				deleteEntities(changeset.getDeleteCiIds(), session);
				if (autocommit) {
					session.save();
				}
			} catch (ReferentialIntegrityException exc) {
				throw new ItemInUseException(exc, "Cannot delete configuration items [%s] because one of the configuration items, or one of their children, is still being referenced", on(',').join(changeset.getDeleteCiIds()));
			}
		}

        private void createEntities(List<ConfigurationItem> entities, Session session) throws RepositoryException {
            verifyEntitiesMeetCreationPreconditions(entities);
            Collections.sort(entities, new SortHierarchyComparator());
            List<NodeWriter> writers = newArrayList();
            for (ConfigurationItem entity : entities) {
                Node node = createNode(entity, session);
                final NodeWriter nodeWriter = new NodeWriter(session, entity, node);
                nodeWriter.writeBasics();
                writers.add(nodeWriter);
            }
            writeEntities(writers);
        }

        private void updateEntities(List<ConfigurationItem> entities, Session session) throws RepositoryException {
            List<Node> nodes = retrieveAndCheckpointNodesToBeUpdated(entities, session);
            for (int i = 0; i < entities.size(); i++) {
                updateNode(entities.get(i), nodes.get(i), session);
            }
        }

        private void deleteEntities(List<String> entityIds, Session session) throws RepositoryException {
            List<Node> nodes = retrieveAndCheckpointNodesToBeDeleted(entityIds, session);
            for (Node node : nodes) {
                deleteNode(node);
            }
        }

        private List<Node> retrieveAndCheckpointNodesToBeUpdated(List<ConfigurationItem> entities, Session session) throws RepositoryException {
            List<Node> nodes = newArrayList();
            for (ConfigurationItem entity : entities) {
                checkNotNull(entity, "entity is null");
                checkNotNull(entity.getId(), "id is null");
                try {
                    final Node node = session.getNode(getAbsolutePathFromId(entity.getId()));
                    checkpoint(node, session);
                    nodes.add(node);
                } catch (PathNotFoundException exc) {
                    throw new NotFoundException("Repository entity [%s] not found", entity.getId());
                }
            }
            return nodes;
        }

        private List<Node> retrieveAndCheckpointNodesToBeDeleted(List<String> ids, Session session) throws RepositoryException {
            List<Node> nodes = newArrayList();
            for (String id : ids) {
                checkNotNull(id, "id is null");
                try {
                    Node node = session.getNode(getAbsolutePathFromId(id));
                    checkpoint(node, session);
                    nodes.add(node);
                } catch (PathNotFoundException ignored) {
                    //ignore
                }
            }
            return nodes;
        }

        private Node createNode(ConfigurationItem entity, Session session) throws RepositoryException {
            validateNodeStoredInCorrectPath(entity, session);
            final Node node = session.getRootNode().addNode(entity.getId());
            int index = node.getIndex();
            if (index != 1) {
                node.remove();
                throw new ItemAlreadyExistsException("Repository entity with ID [%s] already exists", entity.getId());
            }
            setMixins(entity, node);
            return node;
        }

        private void updateNode(ConfigurationItem entity, Node node, Session session) throws RepositoryException {
            new NodeWriter(session, entity, node).write();
        }

        private void deleteNode(Node node) throws RepositoryException {
            node.remove();
        }

        private void checkpoint(Node node, Session session) throws RepositoryException {
            VersionManager versionManager = session.getWorkspace().getVersionManager();
            versionManager.checkpoint(node.getPath());
        }

        private void verifyEntitiesMeetCreationPreconditions(List<ConfigurationItem> cis) {
            for (int i = 0; i < cis.size(); i++) {
                ConfigurationItem ci = cis.get(i);
                checkNotNull(ci, "ci at index %s is null.", i);
                checkNotNull(ci.getId(), "ci at index %s has null id.", i);

                if (ci instanceof SourceArtifact) {
                    checkArgument(((SourceArtifact) ci).getFile() != null, "Artifact %s should have file", ci.getId());
                }
            }
        }

        private void validateNodeStoredInCorrectPath(ConfigurationItem ci, Session session) throws RepositoryException {
            final String id = ci.getId();
            final Type type = ci.getType();
	        checkArgument(DescriptorRegistry.exists(type), "Unknown configuration item type %s", type);

            final Descriptor desc = DescriptorRegistry.getDescriptor(type);

            final String[] pathElements = id.split("/");
            if (desc.getRoot() != NESTED) {
                Checks.checkArgument(pathElements.length == 2 && pathElements[0].equals(desc.getRoot().getRootNodeName()),
                        "Configuration item of type %s cannot be stored at %s. It should be stored under %s", type, id, desc.getRoot().getRootNodeName());
            } else {
                Checks.checkArgument(pathElements.length >= 3,
                        "Configuration item of type %s cannot be stored at %s. It should be stored under a valid parent node", type, id);
                final String parentId = id.substring(0, id.lastIndexOf('/'));
                final Node parentNode = session.getNode(getAbsolutePathFromId(parentId));
                Checks.checkArgument(parentNode != null, "Configuration item of type %s cannot be stored at %s. The parent does not exist", type, id);
                Checks.checkArgument(parentNode.isNodeType(CONFIGURATION_ITEM_NODETYPE_NAME),
                        "Configuration item of type %s cannot be stored at %s. The parent is not a configuration item", type, id);
                final String parentTypeName = parentNode.getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString();
                final Descriptor parentDescriptor = DescriptorRegistry.getDescriptor(parentTypeName);

                // Check parent relation
                for (PropertyDescriptor aPD : desc.getPropertyDescriptors()) {
                    if (aPD.getKind() == CI && aPD.isAsContainment()) {
	                    if (parentDescriptor.isAssignableTo(aPD.getReferencedType())) {
                            return;
                        }
                    }
                }

                // Check child relation
                for (PropertyDescriptor aPD : parentDescriptor.getPropertyDescriptors()) {
                    if (aPD.getKind() == SET_OF_CI && aPD.isAsContainment()) {
	                    if (desc.isAssignableTo(aPD.getReferencedType())) {
                            return;
                        }
                    }
                }

                throw new Checks.IncorrectArgumentException(
                        "Configuration item of type %s cannot be stored at %s. The parent cannot contain configuration items of this type", type, id);
            }
        }

	    private void setMixins(ConfigurationItem entity, Node node)
                throws RepositoryException {
            if (entity instanceof Artifact) {
                node.addMixin(ARTIFACT_NODETYPE_NAME);
                node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
            } else {
                node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
            }
            node.addMixin(NodeType.MIX_REFERENCEABLE);
            node.addMixin(NodeType.MIX_VERSIONABLE);
        }

        private void writeEntities(List<NodeWriter> writers) throws RepositoryException {
            for (NodeWriter writer : writers) {
                writer.write();
            }
        }
    }

    private static class SortHierarchyComparator implements Comparator<ConfigurationItem> {
        @Override
        public int compare(ConfigurationItem o1, ConfigurationItem o2) {
            final int nrSlashes1 = StringUtils.countOccurrencesOf(o1.getId(), "/");
            final int nrSlashes2 = StringUtils.countOccurrencesOf(o2.getId(), "/");
            return nrSlashes1 - nrSlashes2;
        }
    }

    private static class ReferentialIntegrityChecker implements JcrCallback<Object> {

        private final ChangeSet changeset;

        ReferentialIntegrityChecker(ChangeSet changeset) {
            checkNotNull(changeset);
            this.changeset = changeset;
        }

        @Override
        public Object doInJcr(Session session) throws IOException, RepositoryException {
            checkIfAnyOfTheEntitiesToBeCreatedAlreadyExists(session);
            Multimap<String, PropertyRef> deletedEntityRefs = findReferencesToDeletedNodes(session);
            ChangeSetExecutor changeSetExecutor = new ChangeSetExecutor(changeset);
            changeSetExecutor.execute(session, false);
            try {
                checkForRemainingReferencesToDeletedNodes(deletedEntityRefs, session);
            } finally {
                session.refresh(false);
            }
            return null;
        }

        private void checkIfAnyOfTheEntitiesToBeCreatedAlreadyExists(Session session) throws RepositoryException {
            for (ConfigurationItem entity : changeset.getCreateCis()) {
                if (session.nodeExists(getAbsolutePathFromId(entity.getId()))) {
                    throw new ItemAlreadyExistsException("Repository entity %s already exists", entity.getId());
                }
            }
        }

        private Multimap<String, PropertyRef> findReferencesToDeletedNodes(Session session) throws RepositoryException {
            Multimap<String, PropertyRef> deletedEntityRefs = ArrayListMultimap.create();
            for (String deleteNodeId : changeset.getDeleteCiIds()) {
                if (!session.nodeExists(getAbsolutePathFromId(deleteNodeId))) {
                    continue;
                }
                Node node = session.getNode(getAbsolutePathFromId(deleteNodeId));
                PropertyIterator references = node.getReferences();
                while (references.hasNext()) {
                    Property property = references.nextProperty();
                    Node parent = property.getParent();
                    deletedEntityRefs.put(deleteNodeId, new PropertyRef(parent.getPath(), property.getName(), node.getIdentifier()));
                }
            }
            return deletedEntityRefs;
        }

        private void checkForRemainingReferencesToDeletedNodes(Multimap<String, PropertyRef> deletedEntityRefs, Session session) throws RepositoryException {
            for (String deleteNodeId : deletedEntityRefs.keySet()) {
                for (PropertyRef propertyRef : deletedEntityRefs.get(deleteNodeId)) {
                    if (session.nodeExists(propertyRef.owningNode)) {
                        Node owningNode = session.getNode(propertyRef.owningNode);
                        if (owningNode.hasProperty(propertyRef.property)) {
                            Property property = owningNode.getProperty(propertyRef.property);
                            Value[] values = property.isMultiple() ? property.getValues() : new Value[]{property.getValue()};
                            for (Value value : values) {
                                if (value.getString().equals(propertyRef.uuidOfReferencedNode)) {
                                    throw new ItemInUseException("Repository entity %s is still referenced by %s", deleteNodeId, propertyRef.owningNode);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    private static class PropertyRef {
        private String owningNode;
        private String property;
        private String uuidOfReferencedNode;

        PropertyRef(String owningNode, String property, String uuidOfReferencedNode) {
            this.owningNode = owningNode;
            this.property = property;
            this.uuidOfReferencedNode = uuidOfReferencedNode;
        }
    }

	private class Pair<T, U> {
		T left;
		U right;

		private Pair(T left, U right) {
			this.left = left;
			this.right = right;
		}
	}
}
