package com.xebialabs.deployit.jcr.grouping;

import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newTreeSet;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.task.DeploymentTaskInfo;

public class GroupBy {

	private final List<Function<?>> functions;
	private final Map<String, Map<String, Object>> result = newHashMap();
	private final Set<String> propertyNames;
	private final Map<String, Field> fields;

	public GroupBy(Collection<String> properties, Function<?>... functions) {
		this.fields = getFields(properties);
		this.propertyNames = newTreeSet(properties);
		this.functions = functions != null ? Lists.<Function<?>> newArrayList(functions) : Lists.<Function<?>> newArrayList();
	}

	private Map<String, Field> getFields(Collection<String> properties) {
		Map<String, Field> fields = newHashMap();
		for (String property : properties) {
			try {
				Field field = DeploymentTaskInfo.class.getDeclaredField(property);
				field.setAccessible(true);
				fields.put(property, field);
			} catch (NoSuchFieldException e) {
				new IllegalArgumentException(String.format("Property is %s not defined in %s.", property, DeploymentTaskInfo.class.getName()), e);
			}
		}
		return fields;
	}

	public void process(DeploymentTaskInfo task) {
		String key = buildKey(task);
		Map<String, Object> newRow = this.result.get(key);
		if (newRow == null) {
			newRow = newHashMap();
			for (String property : propertyNames) {
				try {
					newRow.put(property, fields.get(property).get(task));
				} catch (IllegalAccessException e) {
					new IllegalArgumentException(String.format("Property is %s not defined in %s.", property, DeploymentTaskInfo.class.getName()), e);
				}
			}
			this.result.put(key, newRow);
		}

		executeFunctions(task, key, newRow);
	}

	private String buildKey(DeploymentTaskInfo task) {
		StringBuilder keyBuilder = new StringBuilder();
		for (String property : propertyNames) {
			try {
				keyBuilder.append(fields.get(property).get(task));
				keyBuilder.append("_");
			} catch (Exception e) {
				new IllegalArgumentException(String.format("Property is %s not defined in %s.", property, DeploymentTaskInfo.class.getName()), e);
			}
		}
		String key = keyBuilder.toString();
		return key;
	}

	private void executeFunctions(DeploymentTaskInfo task, String key, Map<String, Object> newRow) {
		for (Function<?> function : functions) {
			newRow.put(function.getName(), function.invoke(newRow.get(function.getName()), task));
		}
	}

	public Collection<Map<String, Object>> getResult() {
		return result.values();
	}
}
