package com.xebialabs.deployit.jcr;

import static com.xebialabs.deployit.jcr.JcrConstants.ADMIN_PASSWORD;
import static com.xebialabs.deployit.jcr.JcrConstants.ADMIN_USERNAME;

import java.io.IOException;

import javax.jcr.AccessDeniedException;
import javax.jcr.Credentials;
import javax.jcr.LoginException;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.security.AuthenticationFailureException;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.UsernameAndPasswordCredentials;


@Component
public class JcrTemplate {

	private final Repository repository;

    private static final ThreadLocal<UserSession> SESSION_STORE = new ThreadLocal<UserSession>();
    private static final Logger logger = LoggerFactory.getLogger(JcrTemplate.class);

	@Autowired
	public JcrTemplate(final Repository repository) {
		this.repository = repository;
	}

	public final <T> T executeAsAdmin(final JcrCallback<T> callback) {
		return execute(ADMIN_USERNAME, ADMIN_PASSWORD, callback);
	}

	public final <T> T execute(final JcrCallback<T> callback) {
		SecurityContext context = SecurityContextHolder.getContext();
		UsernameAndPasswordCredentials creds = (UsernameAndPasswordCredentials) context.getAuthentication();
		if (creds == null) {
			throw new AuthenticationFailureException();
		}
		return execute(creds.getUsername(), creds.getPassword(), callback);
	}

	public final <T> T execute(final String username, final String password, final JcrCallback<T> callback) throws RuntimeIOException, RuntimeRepositoryException {
        UserSession userSession = SESSION_STORE.get();
        boolean createNewSession = false;
        if(userSession == null) {
        	logger.debug("Did not find an existing session, creating a session for one JCR request.");
        	createNewSession = true;
        } else if(!userSession.session.isLive()) {
        	logger.debug("Existing session is no longer live, creating a session for one JCR request.");
        	createNewSession = true;
        } else if(!userSession.username.equals(username)) {
        	logger.debug("Existing session is for user {} instead of requested user {}, creating a session for one JCR request.", userSession.username, username);
        	createNewSession = true;
        }
        if (createNewSession) {
            try {
                SimpleCredentials credentials = new SimpleCredentials(username, password.toCharArray());
                Session session = repository.login(credentials);
                try {
                    return callback.doInJcr(session);
                } finally {
                    session.logout();
                }
            } catch (AccessDeniedException ade) {
                throw PermissionDeniedException.withMessage("Failed to complete your request.", ade);
            } catch (LoginException le) {
                throw new AuthenticationFailureException(le, "Incorrect credentials for user %s", username);
            } catch (RepositoryException e) {
                throw new RuntimeRepositoryException(e.toString(), e);
            } catch (IOException e) {
                throw new RuntimeIOException(e);
            }
        } else {
            try {
                return callback.doInJcr(userSession.session);
            } catch (IOException e) {
                throw new RuntimeIOException(e);
            } catch (RepositoryException e) {
                throw new RuntimeRepositoryException(e.toString(), e);
            }
        }
	}

    public void login() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (!(authentication instanceof UsernameAndPasswordCredentials)) {
            logger.error("Didn't get a UsernameAndPasswordCredentioals but a: {}", authentication);
            throw new AuthenticationFailureException("User was not yet logged in...");
        }

        UsernameAndPasswordCredentials creds = (UsernameAndPasswordCredentials) authentication;

        logger.debug("Logged in: {}", creds.getUsername());

        Credentials cred = new SimpleCredentials(creds.getUsername(), creds.getPassword().toCharArray());
        try {
            Session session = repository.login(cred);
            SESSION_STORE.set(new UserSession(creds.getUsername(), session));
        } catch (AccessDeniedException ade) {
            throw PermissionDeniedException.withMessage("Failed to complete your request.", ade);
        } catch (LoginException le) {
            throw new AuthenticationFailureException(le, "Incorrect credentials for user %s", creds.getUsername());
        } catch (RepositoryException e) {
            throw new RuntimeRepositoryException(e.toString(), e);
        }
    }

	public boolean isLoggedIn() {
		return SESSION_STORE.get() != null && SESSION_STORE.get().session.isLive();
	}

    public void logout() {
        if (SESSION_STORE.get() != null) {
            SESSION_STORE.get().session.logout();
            SESSION_STORE.remove();
        } else {
            logger.warn("Superfluous logout call found.", new RuntimeException());
        }
    }

    private static class UserSession {
        private String username;
        private Session session;

        private UserSession(String username, Session session) {
            this.username = username;
            this.session = session;
        }
    }
}
