package com.xebialabs.deployit.upgrade;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ListMultimap;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Collections2.transform;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Multimaps.index;
import static com.xebialabs.deployit.jcr.JcrConstants.VERSIONS_NODE_ID;
import static com.xebialabs.deployit.plugin.api.reflect.ReflectionsHolder.getSubTypesOf;
import static com.xebialabs.deployit.server.api.upgrade.Version.VERSION_0;

@Component
public class Upgrader {

    private ApplicationContext context;

    private JcrTemplate jcrTemplate;
    
    private ListMultimap<String, Upgrade> upgradeBeans;

    public @Autowired Upgrader(ApplicationContext context, JcrTemplate jcrTemplate) {
        this.context = context;
        this.jcrTemplate = jcrTemplate;
    }

    @PostConstruct
    public void doUpgrade() {
        upgradeBeans = getUpgrades();

        for (String component : upgradeBeans.keySet()) {
            upgradeComponent(component);
        }
    }


    void upgradeComponent(String component) {
        final Version deployitVersion = readVersionOfComponent(component);

        List<Upgrade> applicableUpgrades = filterUpgrades(upgradeBeans.get(component), deployitVersion);
        if (!applicableUpgrades.isEmpty()) {
            applyUpgrades(applicableUpgrades);
            storeVersionOfComponent(component, applicableUpgrades.get(applicableUpgrades.size() - 1).upgradeVersion());
        }
    }

    void storeVersionOfComponent(final String component, final Version version) {
        jcrTemplate.executeAsAdmin(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws IOException, RepositoryException {
                Node node = session.getNode(VERSIONS_NODE_ID);
                node.setProperty(component, version.getVersion());
                session.save();
                return null;
            }
        });
    }

    Version readVersionOfComponent(final String component) {
        return jcrTemplate.executeAsAdmin(new JcrCallback<Version>() {
            @Override
            public Version doInJcr(Session session) throws IOException, RepositoryException {
                Node node = session.getNode(VERSIONS_NODE_ID);
                if (node.hasProperty(component)) {
                    Property versionProp = node.getProperty(component);
                    return Version.valueOf(component, versionProp.getString());
                } else {
                    return Version.valueOf(component, VERSION_0);
                }
            }
        });
    }

    private void applyUpgrades(List<Upgrade> applicableUpgrades) {
        for (Upgrade applicableUpgrade : applicableUpgrades) {
            if (!applicableUpgrade.doUpgrade()) {
                throw new UpgradeException("Could not perform upgrade %s to upgrade to %s", applicableUpgrade.getClass(), applicableUpgrade.upgradeVersion());
            }
        }
    }

    private List<Upgrade> filterUpgrades(Collection<Upgrade> upgradeBeans, final Version repoVersion) {
        List<Upgrade> applicableUpgrades = newArrayList(filter(upgradeBeans, new Predicate<Upgrade>() {
            @Override
            public boolean apply(Upgrade input) {
                return input.shouldBeApplied(repoVersion);
            }
        }));

        Collections.sort(applicableUpgrades);
        return applicableUpgrades;
    }

    protected ListMultimap<String, Upgrade> getUpgrades() {
        Set<Class<? extends Upgrade>> upgradeClasses = getSubTypesOf(Upgrade.class);
        return index(transform(upgradeClasses, new Function<Class<? extends Upgrade>, Upgrade>() {
            @Override
            public Upgrade apply(Class<? extends Upgrade> input) {
                return context.getAutowireCapableBeanFactory().createBean(input);
            }
        }), new Function<Upgrade, String>() {
            @Override
            public String apply(Upgrade input) {
                return input.upgradeVersion().getComponent();
            }
        });
    }
}
