package com.xebialabs.deployit.service.importer.reader;

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.server.api.importer.ImportingContext;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.importer.ImporterException;
import com.xebialabs.deployit.service.importer.ManifestBasedDarImporter;
import com.xebialabs.overthere.local.LocalFile;
import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.fs.FsSyncException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.Attributes;

import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static java.lang.String.format;

public class DeployableConfigurationItemReader {

	static final String NAMESPACE_SEPARATOR = "/";

	static final String CI = "CI-";

	static final String TYPE_ATTRIBUTE_NAME = "CI-Type";

	static final String NAME_ATTRIBUTE_NAME = "CI-Name";
	private final Map<String, Attributes> entries;

	public DeployableConfigurationItemReader(Map<String, Attributes> entries) {
		this.entries = entries;
	}

	public Deployable readMiddlewareConfiguration(DeploymentPackage deploymentPackage, Map.Entry<String, Attributes> entry) {
		String entryName = entry.getKey().trim();
		Attributes entryAttributes = entry.getValue();

		String configurationItemType = readCIType(entryAttributes, entryName);

		Descriptor descriptor = getDescriptorForConfigurationItem(entryName, configurationItemType);

		Deployable configurationItem = descriptor.newInstance();
		final String baseId = deploymentPackage.getId();
		configurationItem.setId(baseId + "/" + getDeployableNameFromEntryName(entryName));
		fillAttributes(configurationItem, descriptor, entryAttributes, baseId);
		return configurationItem;
	}

	public boolean isMiddlewareResource(Map.Entry<String, Attributes> entry) {
		String configurationItemType = entry.getValue().getValue(TYPE_ATTRIBUTE_NAME);
		String entryName = entry.getKey().trim(); // Is Name: path-relative-to-package-folder/SomeEarFileName.ear

		if (isBlank(configurationItemType)) {
			return false;
		}

		Descriptor configurationItemDescriptor = getDescriptorForConfigurationItem(entryName, configurationItemType);

		// it is a deployable non-artifact
		checkArgument(configurationItemDescriptor.isAssignableTo(Deployable.class), "Configuration item %s of type %s is not Deployable", entryName, configurationItemType);
		return !configurationItemDescriptor.isAssignableTo(DeployableArtifact.class);
	}

	public DeployableArtifact readArtifact(DeploymentPackage deploymentPackage, ImportSource source, Map.Entry<String, Attributes> entry, ImportingContext ctx) {
		String entryName = entry.getKey().trim(); // Is Name: path-relative-to-package-folder/SomeEarFileName.ear
		Attributes entryAttributes = entry.getValue();
		String configurationItemType = readCIType(entryAttributes, entryName);
		Descriptor descriptor = getDescriptorForConfigurationItem(entryName, configurationItemType);
		DeployableArtifact artifact = descriptor.newInstance();

		final String baseId = deploymentPackage.getId();
		artifact.setId(baseId + "/" + getDeployableName(entryName, entryAttributes));
		// Need to wrap in TFile to detect this archivedetector
		TFile sourceArchive = new TFile(source.getFile());
		TFile tempFolder = createTempFolderForImport(ctx, artifact);
		// Prevent going deep in archives by setting archivedetector to NULL.
		TFile artifactFile = new TFile(sourceArchive, entryName, TArchiveDetector.NULL);
		TFile dest = new TFile(tempFolder, artifactFile.getName());
		try {
			artifactFile.cp_r(dest);
			if (sourceArchive.isArchive()) {
				TFile.umount(sourceArchive);
			}
		} catch (FsSyncException e) {
			logger.error("Unable to release resources for archive {}", sourceArchive.getName());
			logger.error("Following exception occurred while trying to release resources: {}", e.getMessage());
		} catch (IOException e) {
			throw new RuntimeIOException(format("Could not copy %s to %s while importing %s", artifactFile, dest, artifact.getId()), e);
		}
		logger.debug("Adding {} {} to deployment package", descriptor.getType(), dest);
		artifact.setFile(LocalFile.valueOf(dest.getFile()));
		fillAttributes(artifact, descriptor, entryAttributes, baseId);
		return artifact;
	}

	private TFile createTempFolderForImport(ImportingContext ctx, Artifact artifact) {
		try {
			String name = "temp-" + artifact.getName();
			TFile tempFolder = new TFile(File.createTempFile(name, "")).rm().mkdir(false);
			ctx.<List<TFile>>getAttribute(ManifestBasedDarImporter.TEMPORARY_FILES).add(tempFolder);
			logger.debug("Created Temporary folder {}", tempFolder);
			return tempFolder;
		} catch (IOException e) {
			throw new RuntimeIOException(e);
		}
	}

	private String getDeployableName(String entryName, Attributes entryAttributes) {
		String name = entryAttributes.getValue(NAME_ATTRIBUTE_NAME);
		if (name != null) {
			return name;
		} else {
			return getDeployableNameFromEntryName(entryName);
		}
	}

	protected String getDeployableNameFromEntryName(String entryName) {
		String name;
		int index = entryName.lastIndexOf(NAMESPACE_SEPARATOR);
		if (index > -1) {
			index++;
			if (index >= entryName.length()) {
				throw new ImporterException("Could not determine artifact Name: %s because entry ends with '/'", entryName);
			}
			name = entryName.substring(index);
		} else {
			name = entryName;
		}
		return name;
	}

	private String readCIType(final Attributes entryAttributes, final String entryName) {
		String configurationItemTypeShortName = entryAttributes.getValue(TYPE_ATTRIBUTE_NAME);
		if (isBlank(configurationItemTypeShortName)) {
			throw new ImporterException("Could not import Name: %s because of missing %s entry in MANIFEST.MF", entryName, TYPE_ATTRIBUTE_NAME);
		}
		return configurationItemTypeShortName;
	}

	private Descriptor getDescriptorForConfigurationItem(String entryName, String configurationItemTypeShortName) {
		Descriptor descriptor = DescriptorRegistry.getDescriptor(configurationItemTypeShortName);
		if (descriptor == null) {
			throw new ImporterException("Could not import Name: %s because %s is not a known CI type", entryName, configurationItemTypeShortName);
		}
		return descriptor;
	}

	private static boolean isBlank(String packageFormatVersion) {
		return Strings.nullToEmpty(packageFormatVersion).trim().isEmpty();
	}

	void fillAttributes(final Deployable entity, final Descriptor configurationItemDescriptor,
	                    final Attributes entryAttributes, final String baseId) {
		for (PropertyDescriptor propertyDescriptor : configurationItemDescriptor.getPropertyDescriptors()) {
			final String name = propertyDescriptor.getName();
			switch (propertyDescriptor.getKind()) {
				case BOOLEAN:
				case INTEGER:
				case STRING:
				case ENUM:
					final String value = entryAttributes.getValue(CI + name);
					propertyDescriptor.set(entity, value);
					break;
				case CI:
					final String mfName = entryAttributes.getValue(CI + name);
					if (mfName == null) {
						continue;
					}
					Attributes referencedAttrs = entries.get(mfName);
					Type ciType = Type.valueOf(readCIType(referencedAttrs, mfName));
					final Descriptor ciDesc = DescriptorRegistry.getDescriptor(ciType);
					Deployable configurationItem = ciDesc.newInstance();
					String ciName = referencedAttrs.getValue(NAME_ATTRIBUTE_NAME);
					if (ciName == null) {
						ciName = mfName;
					}
					configurationItem.setId(IdGenerator.generateId(baseId, ciName));
					propertyDescriptor.set(entity, configurationItem);
					break;
				case SET_OF_STRING:
					Set<String> strings = handleSetOfStrings(propertyDescriptor, entryAttributes);
					if (!strings.isEmpty()) {
						propertyDescriptor.set(entity, strings);
					}
					break;
				case SET_OF_CI:
					Set<Deployable> cis = new HashSet<Deployable>(Collections2.transform(handleSetOfStrings(propertyDescriptor, entryAttributes), new Function<String, Deployable>() {
						@Override
						public Deployable apply(String from) {
							Type ciType = Type.valueOf(readCIType(entries.get(from), from));
							final Descriptor ciDesc = DescriptorRegistry.getDescriptor(ciType);
							Deployable configurationItem = ciDesc.newInstance();
							configurationItem.setId(IdGenerator.generateId(baseId, from));
							return configurationItem;
						}
					}));
					if (!cis.isEmpty()) {
						propertyDescriptor.set(entity, cis);
					}
					break;
				case MAP_STRING_STRING:
					Map<String, String> map = newHashMap();
					String keyStart = (CI + propertyDescriptor.getName() + "-").toLowerCase();
					for (Object attributeKey : entryAttributes.keySet()) {
						String s = attributeKey.toString();
						if (s.toLowerCase().startsWith(keyStart)) {
							String mapKey = s.substring(keyStart.length());
							String mapValue = entryAttributes.getValue(s);
							map.put(mapKey, mapValue);
						}
					}
					propertyDescriptor.set(entity, map);
					break;
				default:
					throw new IllegalStateException("Should not end up here!");

			}
		}
	}

	private static Set<String> handleSetOfStrings(PropertyDescriptor propertyDescriptor, Attributes entryAttributes) {
		String oldStyle = (CI + propertyDescriptor.getName() + "-EntryValue").toLowerCase();
		String key = (CI + propertyDescriptor.getName()).toLowerCase();
		final Set<String> strings = newHashSet();
		for (Map.Entry<Object, Object> entry : entryAttributes.entrySet()) {
			String entryKey = entry.getKey().toString().toLowerCase().trim();
			if (entryKey.startsWith(oldStyle) || entryKey.equals(key)) {
				strings.add(entry.getValue().toString());
			}
		}

		return strings;
	}

	private static final Logger logger = LoggerFactory.getLogger(DeployableConfigurationItemReader.class);

}
