package com.xebialabs.deployit.task;

import com.google.common.collect.Lists;
import com.google.common.io.Files;
import com.xebialabs.deployit.event.EventBus;
import com.xebialabs.deployit.event.EventCallback;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.SecurityCallback;
import com.xebialabs.deployit.security.SecurityTemplate;
import com.xebialabs.deployit.task.Task.State;
import com.xebialabs.deployit.task.Task.TaskStateChangeEvent;
import javassist.util.proxy.ProxyObjectInputStream;
import javassist.util.proxy.ProxyObjectOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.DirectFieldAccessor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

import javax.annotation.PostConstruct;
import java.io.*;
import java.util.*;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.xebialabs.deployit.task.Task.State.*;

public class ArchivingTaskRegistry implements TaskRegistry {

	@Autowired
	private TaskArchive taskArchive;

	// needed to be set in the deserialized tasks
	@Autowired
	private RepositoryService repositoryService;

	private File recoveryFile;

	private int recoveryFileWritingIntervalMillis;

	private Registry registry = new Registry();

	private EventCallback<TaskStateChangeEvent> taskArchivingCallback;

	private Timer recoveryFileWritingTimer;

	@Required
	public void setRecoveryFile(File recoveryFile) {
		this.recoveryFile = recoveryFile;
	}

	@Required
	public void setRecoveryFileWritingIntervalMillis(int recoveryFileWritingIntervalMillis) {
		this.recoveryFileWritingIntervalMillis = recoveryFileWritingIntervalMillis;
	}

	@Override
	public String registerTask(final Task task) {
		final String uuid = UUID.randomUUID().toString();
		task.setId(uuid);
		registry.register(task);
		return uuid;
	}

	@Override
	public Task getTask(final String id) {
		return registry.retrieve(id);
	}

	@Override
	public void cancelTask(String id) {
		Task task = getTask(id);
		if (task.isReadyForExecution()) {
			boolean moveToArchive = false;
			if (task.getState().equals(STOPPED))
				moveToArchive = true;
			
			task.cancel();//Setting status to CANCELLED before persisting.
			
			if (moveToArchive) {
				moveTaskFromRegistryToArchive(task);
			}
			registry.remove(id);
		}
	}

	@Override
	public Collection<Task> getTasks() {
		return registry.tasks();
	}

	@Override
	public List<Task> getIncompleteTasksForUser(String username) {
		List<Task> myTasks = Lists.newArrayList();
		for (Task eachTask : registry.tasks()) {
			// although the registry only contains unfinished tasks, there can
			// be a time-window when the task is finished and is yet to be moved
			// to the task archive, so better check the status
			if ((eachTask.getState().equals(STOPPED) || eachTask.getState().equals(EXECUTING)) && eachTask.getOwner().equals(username)) {
				myTasks.add(eachTask);
			}
		}
		return myTasks;
	}

	@PostConstruct
	public void afterPropertiesSet() {
		loadTasksFromRecoveryFile();
		registerTaskArchivingCallback();
		startRecoveryFileWritingTimer();
	}

	@SuppressWarnings("unchecked")
	private void loadTasksFromRecoveryFile() {
		if (recoveryFile != null && recoveryFile.exists()) {
			try {
				ObjectInputStream recoveryIn = new ProxyObjectInputStream(new FileInputStream(recoveryFile));
				try {
					registry.recover(recoveryIn);
					postprocessTaskRegistryAfterRecovery();
				} finally {
					recoveryIn.close();
				}
			} catch (ClassNotFoundException exc) {
				logger.error("Cannot load tasks from recovery file " + recoveryFile, exc);
			} catch (IOException exc) {
				logger.error("Cannot load tasks from recovery file " + recoveryFile, exc);
			}
		}
	}

	private void postprocessTaskRegistryAfterRecovery() {
		Iterator<Task> tasksIterator = registry.tasks().iterator();
		if (tasksIterator.hasNext()) {
			logger.info("Recovering tasks after server crash");
			while (tasksIterator.hasNext()) {
				Task nextTask = tasksIterator.next();
				State taskState = nextTask.getState();
				if (taskState == PENDING) {
					logger.info("Removing {} task {}", taskState, nextTask.getId());
					tasksIterator.remove();
				} else {
					logger.info("Recovering {} task {}", taskState, nextTask.getId());
					setDepedenciesInTask(nextTask);
					nextTask.processAfterRecovery();
				}
			}
			logger.info("Recovered tasks after server crash");
		}
	}

	private void setDepedenciesInTask(Task eachTask) {
		if (!eachTask.getState().equals(PENDING)) {
			DirectFieldAccessor directFieldAccessor = new DirectFieldAccessor(eachTask);
			if (directFieldAccessor.isWritableProperty("repositoryService"))
				directFieldAccessor.setPropertyValue("repositoryService", repositoryService);
		}
	}

	@Override
	public void destroy() {
		stopRecoveryFileWritingTimer();
		// Make sure that everything is synced up once we're destroyed!
		writeRecoveryFile();
		deregisterTaskArchivingCallback();
	}

	private void registerTaskArchivingCallback() {
		taskArchivingCallback = new EventCallback<TaskStateChangeEvent>() {
			@Override
			public void receive(TaskStateChangeEvent event) {
				Task task = event.getTask();
				if (task.getState().equals(DONE)) {
					moveTaskFromRegistryToArchive(task);
				}
			}

		};
		EventBus.registerForEvent(TaskStateChangeEvent.class, taskArchivingCallback);
	}

	private void deregisterTaskArchivingCallback() {
		EventBus.deregisterForEvent(TaskStateChangeEvent.class, taskArchivingCallback);
	}

	private void startRecoveryFileWritingTimer() {
		recoveryFileWritingTimer = new Timer(ArchivingTaskRegistry.class.getName() + "#recoveryFileWritingTimer", true);
		recoveryFileWritingTimer.schedule(new TimerTask() {
			@Override
			public void run() {
				writeRecoveryFile();
			}
		}, recoveryFileWritingIntervalMillis, recoveryFileWritingIntervalMillis);
	}

	private void stopRecoveryFileWritingTimer() {
		recoveryFileWritingTimer.cancel();
	}

	@SuppressWarnings("deprecation")
    synchronized void writeRecoveryFile() {
		if (recoveryFile != null) {
			File recoveryTemp = new File(recoveryFile.getParentFile(), "recovery.tmp");
			try {
				ObjectOutputStream recoveryOut = new ProxyObjectOutputStream(new FileOutputStream(recoveryTemp));
				try {
					registry.writeRecovery(recoveryOut);
				} finally {
					recoveryOut.close();
				}
				if(recoveryFile.exists()) {
					try {
						Files.deleteRecursively(recoveryFile);
					} catch (IOException e) {
						logger.error("Could not delete {}", recoveryFile, e);
					}
				}
				Files.move(recoveryTemp, recoveryFile);
			} catch (IOException exc) {
				logger.error("Cannot write task registry recovery file", exc);
			}
		}
	}

	private void moveTaskFromRegistryToArchive(final Task task) {
		checkNotNull(task.getOwnerCredentials(), "Cannot archive task " + task.getId() + " because it has no owner");
		SecurityTemplate.executeAs(task.getOwnerCredentials(), new SecurityCallback<Object>() {
			@Override
			public Object doAs() {
				if (task instanceof DeploymentTask) {
					taskArchive.archiveTask((DeploymentTask) task);
				}
				registry.remove(task.getId());
				return null;
			}
		});
	}

	public TaskArchive getTaskArchive() {
		return taskArchive;
	}

	private static final Logger logger = LoggerFactory.getLogger(ArchivingTaskRegistry.class);

}
