/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit;

import com.google.common.base.Strings;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import com.xebialabs.deployit.conversion.StringCoder;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

public class DeployitConfig {

	// ******************
	// Spring Defaults
	// ******************
	public static final String SPRING_CTX_CFG_LOCATION_KEY = "contextConfigLocation";
	public static final String SPRING_CONFIG = "classpath:spring/deployit-context.xml";


	// ******************
	// Jetty HTTP Server Defaults
	// ******************
	public static final String DEFAULT_WEBCONTENT_PACKAGE = "web";
	public static final String DEFAULT_WEB_CONTEXT_ROOT = "/";

	// ******************
	// Property file keys
	// ******************
	public static final String KEY_USE_SSL = "ssl";
	public static final String KEY_SSL_PROTOCOL = "ssl.protocol";
	public static final String KEY_KEYSTORE_PATH = "keystore.path";
	public static final String KEY_KEYSTORE_PASSWORD = "keystore.password";
	public static final String KEY_KEYSTORE_KEYPASSWORD = "keystore.keypassword";
	public static final String KEY_HTTP_PORT = "http.port";
	public static final String KEY_JCR_REPOSITORY_PATH = "jcr.repository.path";
	public static final String KEY_IMPORTABLE_PACKAGES_PATH = "importable.packages.path";
	public static final String KEY_MINIMUM_THREADS = "threads.min";
	public static final String KEY_MAXIMUM_THREADS = "threads.max";
	public static final String KEY_DEFAULT_PREFIX = "default.";

	public static final String KEY_CUSTOM_PREFIX = "custom.";

	public static String CONFIGURATION_FILENAME = "deployit.conf";

	// ******************
	// Fields
	// ******************
	private final File configFile;

	// ******************
	// Mandatory
	// ******************
	protected String keyStorePath;
	protected String keyStorePassword;
	protected String keyStoreKeyPassword;
	protected int httpPort;
	protected String jcrRepositoryPath;
	protected String importablePackagesPath;
	protected String administratorGroupName;
	protected boolean ssl;
	protected String sslProtocol;
	protected int minThreads;

	public int getMinThreads() {
		return minThreads;
	}

	public void setMinThreads(int minThreads) {
		this.minThreads = minThreads;
	}

	public int getMaxThreads() {
		return maxThreads;
	}

	public void setMaxThreads(int maxThreads) {
		this.maxThreads = maxThreads;
	}

	protected int maxThreads;

	private Map<String, String> defaults = new HashMap<String, String>();

	private Map<String, String> customProperties = new HashMap<String, String>();

	private boolean newConfiguration;

	private static DeployitConfig instance;

	public synchronized static DeployitConfig getInstance() {
		if (instance == null) {
			instance = new DeployitConfig();
		}
		return instance;
	}

	public DeployitConfig() {
		configFile = new File("conf", CONFIGURATION_FILENAME);
		this.newConfiguration = true;
	}

	public boolean exists() {
		return configFile.exists();
	}

	public void load() {
		Properties properties = new Properties();
        FileInputStream inStream = null;
		try {
            inStream = Files.newInputStreamSupplier(configFile).getInput();
            properties.load(inStream);
		} catch (IOException e) {
			throw new RuntimeException(e);
		} finally {
            Closeables.closeQuietly(inStream);
        }

		ssl = "true".equalsIgnoreCase(properties.getProperty(KEY_USE_SSL));
		sslProtocol = properties.getProperty(KEY_SSL_PROTOCOL);
		keyStorePath = properties.getProperty(KEY_KEYSTORE_PATH);
		keyStorePassword = properties.getProperty(KEY_KEYSTORE_PASSWORD);
		keyStoreKeyPassword = properties.getProperty(KEY_KEYSTORE_KEYPASSWORD);

		httpPort = parseIntSilent(properties, KEY_HTTP_PORT, httpPort);
		minThreads = parseIntSilent(properties, KEY_MINIMUM_THREADS, minThreads);
		maxThreads = parseIntSilent(properties, KEY_MAXIMUM_THREADS, maxThreads);

		jcrRepositoryPath = properties.getProperty(KEY_JCR_REPOSITORY_PATH);
		importablePackagesPath = properties.getProperty(KEY_IMPORTABLE_PACKAGES_PATH);

		defaults.clear();
		customProperties.clear();
		for (Map.Entry<Object, Object> each : properties.entrySet()) {
			String key = (String) each.getKey();
			String value = (String) each.getValue();
			if (key.startsWith(KEY_DEFAULT_PREFIX)) {
				defaults.put(key.substring(KEY_DEFAULT_PREFIX.length()).toLowerCase(), value);
			}
			if (key.startsWith(KEY_CUSTOM_PREFIX)) {
				customProperties.put(key, value);
			}
		}

		this.newConfiguration = false;
	}

	private int parseIntSilent(Properties properties, String KEY, int defaultValue) {
		try {
			return Integer.parseInt(properties.getProperty(KEY));
		} catch (NumberFormatException nfe) {
			return defaultValue;
		}
	}

	public void save() {
		Properties properties = new Properties();

			properties.setProperty(KEY_USE_SSL, Boolean.toString(isSsl()));
		if (isSsl()) {
			properties.setProperty(KEY_KEYSTORE_PATH, getKeyStorePath());
			properties.setProperty(KEY_KEYSTORE_PASSWORD, this.keyStorePassword);
			properties.setProperty(KEY_KEYSTORE_KEYPASSWORD, this.keyStoreKeyPassword);
			if (getSslProtocol() != null) {
				properties.setProperty(KEY_SSL_PROTOCOL, getSslProtocol());
			}
		}

		properties.setProperty(KEY_HTTP_PORT, Integer.toString(getHttpPort()));
		properties.setProperty(KEY_JCR_REPOSITORY_PATH, getJcrRepositoryPath());
		properties.setProperty(KEY_IMPORTABLE_PACKAGES_PATH, getImportablePackagesPath());

		properties.setProperty(KEY_MINIMUM_THREADS, Integer.toString(minThreads));
		properties.setProperty(KEY_MAXIMUM_THREADS, Integer.toString(maxThreads));

		for (Map.Entry<String, String> each : defaults.entrySet()) {
			properties.put(KEY_DEFAULT_PREFIX + each.getKey(), each.getValue());
		}
		properties.putAll(customProperties);

		FileOutputStream output = null;
		try {
			output = Files.newOutputStreamSupplier(configFile).getOutput();
			properties.store(output, "Deployit configuration file.");
		} catch (IOException e) {
			throw new RuntimeException(e);
		} finally {
			Closeables.closeQuietly(output);
		}
	}

	public boolean isValid() {
		boolean valid = true;
		if (isSsl()) {
			if (isBlank(getKeyStorePath())) {
				System.err.println(KEY_KEYSTORE_PATH + " must be set.");
				valid = false;
			} else if (!new File(getKeyStorePath()).exists()) {
				System.err.println(KEY_KEYSTORE_PATH + " must point to an existing keystore.");
				valid = false;
			}

		}
		if (getHttpPort() < 1) {
			System.err.println(KEY_HTTP_PORT + " is not valid, must be a number larger then 0");
			valid = false;
		}
		if (isBlank(getJcrRepositoryPath())) {
			System.err.println(KEY_JCR_REPOSITORY_PATH + " must be set.");
			valid = false;
		}

		if (isBlank(getImportablePackagesPath())) {
			System.err.println(KEY_IMPORTABLE_PACKAGES_PATH + " must be set.");
			valid = false;
		}

		valid &= validateThatIntIsBiggerThen(KEY_MINIMUM_THREADS, getMinThreads(), 0);
		valid &= validateThatIntIsBiggerThen(KEY_MAXIMUM_THREADS, getMaxThreads(), getMinThreads());

		return valid;
	}

	private boolean isBlank(String keyStorePath) {
		return Strings.nullToEmpty(keyStorePath).trim().isEmpty();
	}

	private boolean validateThatIntIsBiggerThen(String KEY, int intToValidate, int shouldBeBiggerThen) {
		if (intToValidate <= shouldBeBiggerThen) {
			System.err.println(KEY + " is not valid, must be a number larger then " + shouldBeBiggerThen);
			return false;
		} else {
			return true;
		}
	}

	/**
	 * Creates a populated instance of the configuration object. Use this method when configuring the Server or Command Line.
	 * 
	 * @return an instance of the configuration, fully loaded and ready for use.
	 */
	public static DeployitConfig loadForUse() {
		DeployitConfig configuration = getInstance();
		if (configuration.exists()) {
			configuration.load();
			if (!configuration.isValid()) {
				throw new Error("Configuration " + DeployitConfig.CONFIGURATION_FILENAME + " is not valid please run with -setup option.");
			}
		} else {
			throw new Error("Configuration " + DeployitConfig.CONFIGURATION_FILENAME
			        + " does not exist, please run with -setup option before starting the server or command line interface.");
		}
		return configuration;
	}

	public boolean isNewConfiguration() {
		return newConfiguration;
	}

	public String getLocation() {
		return configFile.getAbsolutePath();
	}

	public void setSsl(boolean ssl) {
		this.ssl = ssl;
	}

	public boolean isSsl() {
		return ssl;
	}


	public String getKeyStorePath() {
		return keyStorePath;
	}

	public void setKeyStorePath(String keyStorePath) {
		this.keyStorePath = keyStorePath;
	}

	public String getKeyStorePassword() {
		return StringCoder.decode(keyStorePassword);
	}

	public void setKeyStorePassword(String keyStorePassword) {
		this.keyStorePassword = StringCoder.encode(keyStorePassword);
	}

	public String getKeyStoreKeyPassword() {
		return StringCoder.decode(keyStoreKeyPassword);
	}

	public void setKeyStoreKeyPassword(String keyStoreKeyPassword) {
		this.keyStoreKeyPassword = StringCoder.encode(keyStoreKeyPassword);
	}

	public int getHttpPort() {
		return httpPort;
	}

	public void setHttpPort(int httpPort) {
		this.httpPort = httpPort;
	}

	public String getJcrRepositoryPath() {
		return jcrRepositoryPath;
	}

	public void setJcrRepositoryPath(final String jcrRepositoryPath) {
		this.jcrRepositoryPath = jcrRepositoryPath;
	}


	public String getImportablePackagesPath() {
		return importablePackagesPath;
	}

	public void setImportablePackagesPath(String setImportablePackagesPath) {
		this.importablePackagesPath = setImportablePackagesPath;
	}

	public Map<String, String> getDefaults() {
		return defaults;
	}

	public void setDefaults(Map<String, String> defaults) {
		this.defaults = defaults;
	}

	public Map<String, String> getCustomProperties() {
		return customProperties;
	}

	public void setCustomProperties(Map<String, String> customProperties) {
		this.customProperties = customProperties;
	}

	public String getAdministratorGroupName() {
		return administratorGroupName;
	}

	public void setAdministratorGroupName(String administratorGroupName) {
		this.administratorGroupName = administratorGroupName;
	}

	public String getSslProtocol() {
		return sslProtocol;
	}

	public void setSslProtocol(String sslProtocol) {
		this.sslProtocol = sslProtocol;
	}

}