package com.xebialabs.deployit.service.deployment;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Collections2;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static java.lang.String.format;

@Component
public class DeployedService {
    private PojoConverter pojoConverter;
    private DeployedGenerator deployedGenerator;

    public @Autowired DeployedService(PojoConverter pojoConverter, DeployedGenerator deployedGenerator) {
        this.pojoConverter = pojoConverter;
        this.deployedGenerator = deployedGenerator;
    }

    public ListMultimap<Boolean, ConfigurationItemEntity> generateAllDeployeds(ConfigurationItemEntity deploymentPackageEntity, ConfigurationItemEntity environmentEntity) {
        final PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
        try {
            final DeploymentPackage pkg = pojoConverterContext.toPojo(deploymentPackageEntity);
	        final Environment env = pojoConverterContext.toPojo(environmentEntity);
            Set<Container> containers = env.getMembers();
	        ConsolidatedDictionary dictionary = ConsolidatedDictionary.create(env.getDictionaries());
	        List<Deployed> validDeployeds = generateDeployedsOfType(pkg.getDeployables(), containers, null, dictionary);
            if (validDeployeds.isEmpty()) {
                throw new IllegalArgumentException("Could not generate any deployeds for " + deploymentPackageEntity.getId() + " and " + environmentEntity.getId());
            }
            return createValidDeployedMap(validDeployeds, Lists.<Deployed>newArrayList(), pojoConverterContext);
        } finally {
            pojoConverterContext.destroy();
        }
    }

	public ListMultimap<Boolean, ConfigurationItemEntity> generateSelectedDeployeds(List<ConfigurationItemEntity> deployableEntities, ConfigurationItemEntity environmentEntity) {
        final PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
        try {
	        final Environment env = pojoConverterContext.toPojo(environmentEntity);
            Set<Container> containers = env.getMembers();
	        ConsolidatedDictionary dictionary = ConsolidatedDictionary.create(env.getDictionaries());

            List<Deployable> deployables = Lists.transform(deployableEntities, new Function<ConfigurationItemEntity, Deployable>() {
                @Override
                public Deployable apply(ConfigurationItemEntity from) {
                    final ConfigurationItem deployable = pojoConverterContext.toPojo(from);
                    checkArgument(deployable instanceof Deployable, "The entity %s is not a deployable", from.getId());
                    return (Deployable) deployable;
                }
            });

            return createValidDeployedMap(generateDeployedsOfType(deployables, containers, null, dictionary), Lists.<Deployed>newArrayList(), pojoConverterContext);
        } finally {
            pojoConverterContext.destroy();
        }
    }

	public ListMultimap<Boolean, ConfigurationItemEntity> generateSelectedDeployed(ConfigurationItemEntity deployableEntity, ConfigurationItemEntity containerEntity, String deployedTypeString, ConfigurationItemEntity environmentEntity) {
		final PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
		try {
			Type deployedType = null;
			if (deployedTypeString != null && !deployedTypeString.trim().isEmpty()) {
				deployedType = Type.valueOf(deployedTypeString);
			}

			Deployable deployable = pojoConverterContext.toPojo(deployableEntity);
			Container container = pojoConverterContext.toPojo(containerEntity);
			Environment environment = pojoConverterContext.toPojo(environmentEntity);
			ConsolidatedDictionary dictionary = ConsolidatedDictionary.create(environment.getDictionaries());
			return createValidDeployedMap(generateDeployedsOfType(newArrayList(deployable), newArrayList(container), deployedType, dictionary), Lists.<Deployed>newArrayList(), pojoConverterContext);
		} finally {
		    pojoConverterContext.destroy();
		}

	}

    public ListMultimap<Boolean, ConfigurationItemEntity> generateUpgradedDeployeds(ConfigurationItemEntity newSourceEntity, ConfigurationItemEntity deploymentEntity) {
        final PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
        try {
            final DeployedApplication deployment = pojoConverterContext.toPojo(deploymentEntity);
	        final Environment environment = deployment.getEnvironment();
	        ConsolidatedDictionary dictionary = ConsolidatedDictionary.create(environment.getDictionaries());
	        final DeploymentPackage newPackage = pojoConverterContext.toPojo(newSourceEntity);

            List<Deployed> upgradeDeployeds = newArrayList();
            List<Deployed> oldDeployeds = newArrayList();
            for (final Deployed deployed : deployment.getDeployeds()) {
                Collection<Deployable> filtered = Collections2.filter(newPackage.getDeployables(), new Predicate<Deployable>() {
                    @Override
                    public boolean apply(Deployable input) {
                        return isSimilar(input, deployed.getDeployable());
                    }
                });

                if (filtered.size() == 0) {
                    oldDeployeds.add(deployedGenerator.generateUpgradedDeployed(deployed.getDeployable(), deployed, dictionary));
                } else if (filtered.size() == 1) {
                    upgradeDeployeds.add(deployedGenerator.generateUpgradedDeployed(filtered.iterator().next(), deployed, dictionary));
                } else {
                    throw new IllegalArgumentException(format("More than 1 applicable deployable found in new package for %s, will not fly.", deployed));
                }
            }

            return createValidDeployedMap(upgradeDeployeds, oldDeployeds, pojoConverterContext);
        } finally {
            pojoConverterContext.destroy();
        }
    }

    private ListMultimap<Boolean, ConfigurationItemEntity> createValidDeployedMap(List<Deployed> validDeployeds, List<Deployed> invalidDeployeds, PojoConverter.Context pojoConverterContext) {
        ListMultimap<Boolean, ConfigurationItemEntity> deployedsMap = ArrayListMultimap.create();
        deployedsMap.putAll(Boolean.TRUE, convertDeployeds(validDeployeds, pojoConverterContext));
        deployedsMap.putAll(Boolean.FALSE, convertDeployeds(invalidDeployeds, pojoConverterContext));

        return deployedsMap;
    }

    private boolean isSimilar(Deployable newDeployable, Deployable deployable) {
        return newDeployable.getType().equals(deployable.getType()) && newDeployable.getName().equals(deployable.getName());
    }

    private List<Deployed> generateDeployedsOfType(Collection<Deployable> deployables, Collection<Container> containers, Type deployedType, ConsolidatedDictionary dictionary) {
        List<Deployed> deployeds = newArrayList();
        for (Deployable deployable : deployables) {
            for (Container container : containers) {
                if (deployedType != null) {
                    if (deployedGenerator.findDeployedTypesForDeployableAndContainerTypes(deployable.getType(), container.getType()).contains(deployedType)) {
                        deployeds.add(deployedGenerator.generateDeployed(deployable, container, deployedType, dictionary));
                    }
                } else {
                    Deployed deployed = deployedGenerator.generateMostSpecificDeployed(deployable, container, dictionary);
                    if (deployed != null) {
                        deployeds.add(deployed);
                    }
                }
            }
        }
        return deployeds;
    }

    private List<ConfigurationItemEntity> convertDeployeds(List<Deployed> deployeds, final PojoConverter.Context pojoConverterContext) {
        return newArrayList(Lists.transform(deployeds, new Function<Deployed, ConfigurationItemEntity>() {
            @Override
            public ConfigurationItemEntity apply(Deployed from) {
                return pojoConverterContext.toEntity(from);
            }
        }));
    }
}
