package com.xebialabs.deployit.service.deployment;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import org.springframework.stereotype.Component;

import java.util.*;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.service.replacement.MustachePlaceholderScanner.hasPlaceholders;
import static java.lang.String.format;

@Component
public class DeployedGenerator {


	public Type findMostSpecificDeployedTypeForDeployableAndContainerTypes(final Type deployableType, final Type containerType) {
        List<Type> deployedTypesForDeployableAndContainer = findDeployedTypesForDeployableAndContainerTypes(deployableType, containerType);

        if (deployedTypesForDeployableAndContainer.isEmpty()) {
            return null;
        }
        if (deployedTypesForDeployableAndContainer.size() == 1) {
            return deployedTypesForDeployableAndContainer.get(0);
        }

        List<Type> copy = filterMostSpecific(deployedTypesForDeployableAndContainer);

        if (copy.size() == 1) {
            return copy.get(0);
        }

        throw new IllegalArgumentException(format("Found %s for %s and %s, expected only 1", copy, deployableType, containerType));
    }

    List<Type> filterMostSpecific(final List<Type> deployedTypesForDeployableAndContainer) {
        List<Type> copy = newArrayList(deployedTypesForDeployableAndContainer);

        for (Type currentType : deployedTypesForDeployableAndContainer) {
            Descriptor currentDescriptor = DescriptorRegistry.getDescriptor(currentType);

            for (Iterator<Type> copyIt = copy.iterator(); copyIt.hasNext();) {
                Type comparedType = copyIt.next();
                Descriptor comparedDescriptor = DescriptorRegistry.getDescriptor(comparedType);

                boolean deployableEquals = comparedDescriptor.getDeployableType().equals(currentDescriptor.getDeployableType());
                boolean containerEquals = comparedDescriptor.getContainerType().equals(currentDescriptor.getContainerType());
                boolean containerSupertype = isSubType(comparedDescriptor.getContainerType(), currentDescriptor.getContainerType());
                boolean deployableSupertype = isSubType(comparedDescriptor.getDeployableType(), currentDescriptor.getDeployableType());

                boolean isSupertype = isSubType(comparedType, currentType);

                if ((containerSupertype && deployableEquals)
                        || (deployableSupertype && containerEquals)
                        || (deployableSupertype && containerSupertype)
                        || isSupertype) {
                    copyIt.remove();
                }
            }
        }
        return copy;
    }

    private boolean isSubType(final Type superType, final Type subType) {
        return DescriptorRegistry.getSubtypes(superType).contains(subType);
    }

    public List<Type> findDeployedTypesForDeployableAndContainerTypes(final Type deployableType, final Type containerType) {
        Collection<Type> deployedTypes = DescriptorRegistry.getSubtypes(Type.valueOf(Deployed.class));
        final Descriptor deployableDesc = DescriptorRegistry.getDescriptor(deployableType);
        final Descriptor containerDesc = DescriptorRegistry.getDescriptor(containerType);
        return newArrayList(Collections2.filter(deployedTypes, new Predicate<Type>() {
            @Override
            public boolean apply(Type input) {
                Descriptor deployedDesc = DescriptorRegistry.getDescriptor(input);
                return !deployedDesc.isVirtual() && deployableDesc.isAssignableTo(deployedDesc.getDeployableType()) && containerDesc.isAssignableTo(deployedDesc.getContainerType());
            }
        }));
    }

    public Deployed<?, ?> generateMostSpecificDeployed(final Deployable deployable, final Container container, final ConsolidatedDictionary dictionary) {
        Type deployedType = findMostSpecificDeployedTypeForDeployableAndContainerTypes(deployable.getType(), container.getType());
        if (deployedType != null) {
            return generateDeployed(deployable, container, deployedType, dictionary);
        }
        return null;
    }

    public Deployed<?, ?> generateDeployed(final Deployable deployable, final Container container, final Type deployedType, final ConsolidatedDictionary dictionary) {
        Deployed<Deployable, Container> deployed = createDeployed(deployable, container, deployedType);
        setPropertyPlaceholders(deployed, deployable, dictionary);
		setFilePlaceholders(deployed, deployable, dictionary);
        return deployed;
    }

	private Deployed<Deployable, Container> createDeployed(final Deployable deployable, final Container container, final Type deployedType) {
	    Descriptor deployedDesc = DescriptorRegistry.getDescriptor(deployedType);

	    Deployed<Deployable, Container> deployed = deployedDesc.newInstance();
	    deployed.setDeployable(deployable);
        deployed.setContainer(container);
        IdGenerator.generateId(deployed);

        return deployed;
    }

	private void setPropertyPlaceholders(Deployed<Deployable, Container> deployed, final Deployable deployable, final ConsolidatedDictionary dictionary) {
        Descriptor deployedDesc = DescriptorRegistry.getDescriptor(deployed.getType());
	    Descriptor deployableDesc = DescriptorRegistry.getDescriptor(deployable.getType());

	    for (PropertyDescriptor deployedPropDesc : deployedDesc.getPropertyDescriptors()) {
	        String name = deployedPropDesc.getName();
	        if (name.equals("placeholders")) {
	        	continue;
	        }

	        PropertyDescriptor deployablePropDesc = deployableDesc.getPropertyDescriptor(name);
		    if (deployablePropDesc != null) {
			    Object value = deployablePropDesc.get(deployable);
			    // TODO replace deployedPropDesc.toString with getFqn();
			    if (value == null && dictionary.containsKey(deployedPropDesc.toString())) {
				    value = dictionary.get(deployedPropDesc.toString());
				    deployedPropDesc.set(deployed, value);
			    } else {
					try {
						value = dictionary.resolve(value);
						deployedPropDesc.set(deployed, value);
					} catch (DictionaryValueException exc) {
						logger.error("Could not resolve dictionary keys for property " + deployablePropDesc + " on " + deployed + ". Property will be left empty.", exc);
					} catch (RuntimeException exc) {
						logger.error("Could not convert (resolved) value to correct type for property " + deployablePropDesc + " on " + deployed + ". Property will be left empty.", exc);
					}
			    }
		    }
        }
    }

	private void setFilePlaceholders(final Deployed<Deployable, Container> deployed, final Deployable deployable, final ConsolidatedDictionary dictionary) {
	    Descriptor deployedDesc = DescriptorRegistry.getDescriptor(deployed.getType());
		Descriptor deployableDesc = DescriptorRegistry.getDescriptor(deployable.getType());

		PropertyDescriptor deployedPlaceholdersDesc = deployedDesc.getPropertyDescriptor("placeholders");
		if (deployedPlaceholdersDesc != null) {
	        PropertyDescriptor deployableplaceholders = deployableDesc.getPropertyDescriptor("placeholders");
	        @SuppressWarnings("unchecked")
            Set<String> placeholders = (Set<String>) deployableplaceholders.get(deployable);

	        deployedPlaceholdersDesc.set(deployed, resolveFilePlaceholders(Maps.<String, String>newHashMap(), placeholders, dictionary));
		}
    }

	public Deployed<? ,?> generateUpgradedDeployed(final Deployable newDeployable, final Deployed<?, ?> existingDeployed, final ConsolidatedDictionary dictionary) {
        Deployed<?, ?> upgradedDeployed = createUpgradedDeployed(newDeployable, existingDeployed);
        setUpgradedPropertyPlaceholders(upgradedDeployed, newDeployable, existingDeployed, dictionary);
		setUpgradedFilePlaceholders(upgradedDeployed, newDeployable, existingDeployed, dictionary);
        return upgradedDeployed;
    }

	private Deployed<?, ?> createUpgradedDeployed(final Deployable newDeployable, final Deployed<?, ?> existingDeployed) {
	    Descriptor deployedDesc = DescriptorRegistry.getDescriptor(existingDeployed.getType());
        Type deployableType = deployedDesc.getDeployableType();
        Type newDeployableType = newDeployable.getType();
		checkArgument(deployableType.equals(newDeployableType) || DescriptorRegistry.getSubtypes(deployableType).contains(newDeployableType),
		        "The new Deployable type %s should be assignable to %s", newDeployableType, deployableType);

        Deployed<Deployable, Container> upgradedDeployed = deployedDesc.newInstance();
        upgradedDeployed.setDeployable(newDeployable);
        upgradedDeployed.setId(existingDeployed.getId());

        return upgradedDeployed;
    }

	private void setUpgradedPropertyPlaceholders(final Deployed<?, ?> upgradedDeployed, final Deployable newDeployable, final Deployed<?, ?> existingDeployed, final ConsolidatedDictionary dictionary) {
        Descriptor deployedDesc = DescriptorRegistry.getDescriptor(existingDeployed.getType());
        Descriptor deployableDesc = DescriptorRegistry.getDescriptor(deployedDesc.getDeployableType());

        for (PropertyDescriptor deployedPropDesc : deployedDesc.getPropertyDescriptors()) {
            String name = deployedPropDesc.getName();
	        if (name.equals("placeholders") || name.equals("deployable")) {
		        continue;
	        }

	        Object oldDeployedValue = deployedPropDesc.get(existingDeployed);
            Object valueToSet = oldDeployedValue;

            PropertyDescriptor deployablePropDesc = deployableDesc.getPropertyDescriptor(name);
            if (deployablePropDesc != null) {
	            Object oldDeployableValue = deployablePropDesc.get(existingDeployed.getDeployable());
                if ((oldDeployableValue != null) && (oldDeployableValue.equals(oldDeployedValue) || hasPlaceholders(oldDeployableValue.toString()))) {
                    valueToSet = deployablePropDesc.get(newDeployable);
                }

	            try {
		            valueToSet = dictionary.resolve(valueToSet);
	            } catch (DictionaryValueException exc) {
		            valueToSet = oldDeployedValue;
		            logger.error("Could not resolve dictionary keys for property " + deployedPropDesc + " of " + upgradedDeployed + ". Property will be set to old value.", exc);
	            }
            }

            try {
                deployedPropDesc.set(upgradedDeployed, valueToSet);
	        } catch (RuntimeException exc) {
		        logger.error("Could not convert (resolved) value to correct type for property " + deployedPropDesc + " of " + upgradedDeployed + ". Property will be left empty.", exc);
	        }
        }
    }

	private void setUpgradedFilePlaceholders(final Deployed<?, ?> upgradedDeployed, final Deployable newDeployable, final Deployed<?, ?> existingDeployed, final ConsolidatedDictionary dictionary) {
		Descriptor deployedDesc = DescriptorRegistry.getDescriptor(upgradedDeployed.getType());
		Descriptor deployableDesc = DescriptorRegistry.getDescriptor(newDeployable.getType());

		PropertyDescriptor deployedPlaceholdersDesc = deployedDesc.getPropertyDescriptor("placeholders");
		if (deployedPlaceholdersDesc != null) {
			@SuppressWarnings("unchecked")
            Map<String, String> existingDeployedPlaceholders = (Map<String, String>) deployedPlaceholdersDesc.get(existingDeployed);
			@SuppressWarnings("unchecked")
            Set<String> newDeployablePlaceholders = (Set<String>) deployableDesc.getPropertyDescriptor("placeholders").get(newDeployable);

			deployedPlaceholdersDesc.set(upgradedDeployed, resolveFilePlaceholders(existingDeployedPlaceholders, newDeployablePlaceholders, dictionary));
		}
	}

	private Map<String, String> resolveFilePlaceholders(final Map<String, String> existingDeployedPlaceholders, final Set<String> deployablePlaceholders, final ConsolidatedDictionary dictionary) {
		Map<String, String> newDeployedPlaceholders = newHashMap();
		for (String deployablePlaceholder : deployablePlaceholders) {
			String placeholderValue;
			try {
				placeholderValue = (String) dictionary.resolve("{{" + deployablePlaceholder + "}}");
			} catch (DictionaryValueException ignored) {
				if (existingDeployedPlaceholders.containsKey(deployablePlaceholder)) {
					placeholderValue = existingDeployedPlaceholders.get(deployablePlaceholder);
				} else {
					placeholderValue = "";
				}
			}
			newDeployedPlaceholders.put(deployablePlaceholder, placeholderValue);
		}
		return newDeployedPlaceholders;
	}

	private static final Logger logger = LoggerFactory.getLogger(DeployedGenerator.class);
}
