package com.xebialabs.deployit.security;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.security.permission.Permission;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import java.io.IOException;

public class JcrAuthenticationProvider implements AuthenticationProvider {

	@Autowired
	private JcrTemplate jcrTemplate;

	@Override
	public Authentication authenticate(final Authentication token) throws AuthenticationException {
		logger.debug("Authenticating for Deployit: {}", token.getPrincipal());
		if (!(token instanceof UsernamePasswordAuthenticationToken)) {
			return null;
		}
		final String username = token.getPrincipal().toString();
		final String password = token.getCredentials().toString();

		checkPrincipalAndCredentialsWithJcr(username, password);

		return new UsernameAndPasswordCredentials(username, password);
	}

	private void checkPrincipalAndCredentialsWithJcr(final String username, final String password) {
		if (Strings.nullToEmpty(username).trim().isEmpty()) {
			throw new BadCredentialsException("Cannot authenticate with empty username");
		}

		// execute an empty callback to check that the user can log in to JCR
		try {
			jcrTemplate.execute(username, password, new JcrCallback<Object>() {
				@Override
				public Object doInJcr(Session session) throws IOException, RepositoryException {
					return null;
				}
			});
			SecurityTemplate.executeAs(username, password, new SecurityCallback<Object>() {
				@Override
				public Object doAs() {
					if (Permission.LOGIN.getPermissionHandler().hasPermission(Lists.<String>newArrayList())) {
						return null;
					} else {
						throw new AuthenticationFailureException("Login permission not granted.");
					}
				}
			});
		} catch (AuthenticationFailureException exc) {
			throw new BadCredentialsException("Cannot authenticate " + username, exc);
		}
	}

	@Override
	public boolean supports(final Class<? extends Object> authentication) {
		return authentication.isAssignableFrom(UsernamePasswordAuthenticationToken.class);
	}

	private static final Logger logger = LoggerFactory.getLogger(JcrAuthenticationProvider.class);
}
