package com.xebialabs.deployit.plugin;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.io.ByteStreams.newInputStreamSupplier;
import static com.xebialabs.deployit.plugin.Exploder.explode;
import static java.lang.Math.abs;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.io.ByteStreams;
import com.google.common.io.InputSupplier;
import com.google.common.io.OutputSupplier;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FileArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.PlaceholderReplacer;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.repository.ArtifactEntity;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.service.replacement.MustachePlaceholderReplacer;
import com.xebialabs.deployit.util.GuavaFiles;
import com.xebialabs.deployit.util.PasswordObfuscator;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.local.LocalFile;

public class PojoConverter implements InitializingBean {

	public static final OverthereFile NO_ARTIFACT_DATA_LOCATION = LocalFile.valueOf(new File("no-artifact-data"));

	private final RepositoryService repositoryService;

	private File workDir;

	@Autowired
	public PojoConverter(final RepositoryService repositoryService) {
		this.repositoryService = repositoryService;
	}

	@Override
	public void afterPropertiesSet() {
		if (!workDir.mkdirs() && !workDir.exists()) {
			throw new RuntimeIOException("Cannot create work dir " + workDir);
		}
	}

	public Context getContext() {
		return new Context();
	}

	public class Context {
		private Map<String, ConfigurationItemEntity> entitiesQueuedForConversion = newHashMap();
		private List<File> artifactDataWorkDirs = newArrayList();
		private Map<String, ConfigurationItem> convertedPojos = newHashMap();

		private Context() {
		}

        public <F extends ConfigurationItemEntity, G> Collection<G> toPojo(final Collection<F> entities) {
            for (ConfigurationItemEntity entity : entities) {
                entitiesQueuedForConversion.put(entity.getId(), entity);
            }

            try {
                Collection<G> result = newArrayList();
                for (ConfigurationItemEntity entity : entities) {
                    result.add(this.<G>toPojo(entity));
                }
                return result;
            } finally {
                entitiesQueuedForConversion.clear();
            }
        }

        private ConfigurationItem newPojoInstance(ConfigurationItemEntity item) {
            final Descriptor descriptor = DescriptorRegistry.getDescriptor(item.getType());
            ConfigurationItem pojo = descriptor.newInstance();
            fillId(pojo, item);
            return pojo;
        }

		@SuppressWarnings("unchecked")
		public <T> T toPojo(ConfigurationItemEntity item) {
			if (convertedPojos.containsKey(item.getId())) {
				logger.debug("Returning previously generated POJO for configuration item {}", item.getId());
				return (T) convertedPojos.get(item.getId());
			}

			logger.debug("Reading {}", item.getId());
			ConfigurationItem pojo = newPojoInstance(item);
            storeReferenceToPojo(pojo);
			fillFields(pojo, item, DescriptorRegistry.getDescriptor(item.getType()));
			if (item instanceof ArtifactEntity) {
				initSourceArtifact(pojo, (ArtifactEntity) item);
			}
			if (pojo instanceof DerivedArtifact<?>) {
				initDerivedArtifact((DerivedArtifact<? extends SourceArtifact>) pojo);
			}

			return (T) pojo;
		}

		private void storeReferenceToPojo(final ConfigurationItem pojo) {
			// The created object is stored in the convertedPojos map right away to make sure that no StackOverflowError occurs when a CI refers to itself
			// either directly or indirectly
			convertedPojos.put(pojo.getId(), pojo);
		}

		private void fillId(final ConfigurationItem pojo, final ConfigurationItemEntity item) {
			pojo.setId(item.getId());
		}

		private void fillFields(final ConfigurationItem pojo, final RepositoryObjectEntity item, final Descriptor descriptor) {
			for (PropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {
				fillProperty(pojo, item, propertyDescriptor);
			}
		}


		private void fillProperty(final ConfigurationItem pojo, final RepositoryObjectEntity entity, final PropertyDescriptor propertyDescriptor) {
			final String propertyName = propertyDescriptor.getName();
			final Object itemValue = entity.getValue(propertyName);
			setPropertyInPojo(pojo, itemValue, propertyDescriptor);
		}

		private <T extends ConfigurationItem> void setPropertyInPojo(final T pojo, final Object pojoValue, final PropertyDescriptor propertyDescriptor) {
            if (pojoValue == null)
                return;

            logger.debug("Loading {} of {} with value {}", new Object[] { propertyDescriptor.getName(), pojo, pojoValue });
			Object value = convertValueForPojo(pojoValue, propertyDescriptor);
			propertyDescriptor.set(pojo, value);
		}

		@SuppressWarnings("unchecked")
		private Object convertValueForPojo(Object pojoValue, PropertyDescriptor propertyDescriptor) {
            Object value;

            switch (propertyDescriptor.getKind()) {
            case STRING:
                if (propertyDescriptor.isPassword()) {
                    value = PasswordObfuscator.decrypt(pojoValue.toString());
	                break;
                } // Fall-through to next case.
            case BOOLEAN:
            case INTEGER:
            case ENUM:
                value = pojoValue;
                break;
            case SET_OF_STRING:
                value = handleSetOfString(pojoValue);
                break;
            case CI:
                value = handleConfigurationItem((String) pojoValue);
                break;
            case SET_OF_CI:
            	value = handleSetofConfigurationItems((Collection<String>) pojoValue);
                break;
            case MAP_STRING_STRING:
	            value = pojoValue != null ? newHashMap((Map<String, String>) pojoValue) : null;
	            break;
            default:
                throw new IllegalStateException("Should not end up here!");
            }
            return value;
        }

        @SuppressWarnings("unchecked")
		private Set<String> handleSetOfString(Object pojoValue) {
			if (pojoValue instanceof Set) {
				return (Set<String>) pojoValue;
			} else if (pojoValue instanceof Collection) {
				return newHashSet((Collection<String>) pojoValue);
			} else {
				throw new IllegalStateException("Did not get a Set of String or other Collection type.");
			}
		}

		private Object handleSetofConfigurationItems(final Collection<String> ids) {
			Set<Object> result = Sets.newHashSet();
			for (String id : ids) {
				result.add(handleConfigurationItem(id));
			}
			return result;
		}

		private Object handleConfigurationItem(final String id) {
			// Even though toPojo will check the convertedPojos map again, we save a call to readFully by checking the map early
			if (convertedPojos.containsKey(id)) {
				return convertedPojos.get(id);
			}

            final ConfigurationItemEntity ci;
            if (entitiesQueuedForConversion.containsKey(id)) {
                ci = entitiesQueuedForConversion.get(id);
            } else {
                ci = repositoryService.readFully(id);
            }

			return toPojo(ci);
		}

		private void initSourceArtifact(final ConfigurationItem o, final ArtifactEntity item) {
			if (item.getData() == null) {
				logger.warn("Not filling in the artifact data, because it wasn't set.");
				return;
			}

			if (o instanceof FolderArtifact) {
				FolderArtifact da = (FolderArtifact) o;
				da.setFile(LocalFile.valueOf(writeArtifactDataToWorkFolder(item)));
			} else if (o instanceof FileArtifact) {
				FileArtifact da = (FileArtifact) o;
				da.setFile(LocalFile.valueOf(writeArtifactDataToWorkFile(item)));
			} else {
				throw new IllegalArgumentException("Unknown subclass of " + Artifact.class + ": " + o.getClass());
			}
		}

		private void initDerivedArtifact(DerivedArtifact<? extends SourceArtifact> da) {
			if (da.getSourceArtifact() != null) {
				da.initFile(new PlaceholderReplacer() {
					@Override
					public void replace(Reader in, Writer out, Map<String, String> resolution) {
						new MustachePlaceholderReplacer(resolution).replace(in, out);
					}
				});
				if (da.getFile() == null) {
					// This pojo has not been converted.
					convertedPojos.remove(da.getId());
					throw new Checks.IncorrectArgumentException("After initialization DeployedArtifact %s has not set a file, cannot deploy.", da.getId());
				}
			}
		}

		private File writeArtifactDataToWorkFile(final ArtifactEntity item) {
			try {
				logger.debug("Writing data for artifact {} to temporary file in work directory {}", item.getId(), workDir);
                final File artifactDataWorkFile = getArtifactDataWorkFile(item);
				ByteStreams.copy(item.getData(), new OutputSupplier<OutputStream>() {
                    public OutputStream getOutput() throws IOException {
	                    return new FileOutputStream(artifactDataWorkFile);
                    }
				});
				logger.debug("Wrote data for artifact {} to file {} in work directory", item.getId(), artifactDataWorkFile);
				return artifactDataWorkFile;
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot write artifact data to temporary file for artifact " + item.getId(), exc);
			}
		}

		private File writeArtifactDataToWorkFolder(final ArtifactEntity item) {
			try {
				logger.debug("Exploding folder artifact {} to temporary directory work directory {}", item.getId(), workDir);
                final File artifactDataWorkDir = getArtifactDataWorkFile(item);
                explode(item.getData(), artifactDataWorkDir);
				logger.debug("Exploded folder artifact {} to directory {} in work directory", item.getId(), artifactDataWorkDir);
				return artifactDataWorkDir;
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot write artifact data to temporary folder for artifact " + item.getId(), exc);
			}
		}

		private File getArtifactDataWorkFile(final ArtifactEntity item) throws IOException {
	        String baseDirName = item.getId().replace('/', '_');
	        Random r = new Random();
	        for(int tries = 0; tries < 100; tries++) {
				File artifactDataWorkDir = new File(workDir, baseDirName + abs(r.nextLong()));
				if(artifactDataWorkDir.exists()) {
					continue;
				}

				if(!artifactDataWorkDir.mkdir()) {
		        	throw new IOException("Cannot create directory " + artifactDataWorkDir);
		        }
				
				artifactDataWorkDirs.add(artifactDataWorkDir);
		        return new File(artifactDataWorkDir, item.getFilename());
	        }
	        throw new IllegalStateException("Cannot generate work dir for " + item);
        }

		public void destroy() {
			deleteArtifactDataWorkDirs();
			setArtifactFileToPointToNonExistentFile();
		}

		private void deleteArtifactDataWorkDirs() {
	        for(File d : artifactDataWorkDirs) {
				try {
	                GuavaFiles.deleteRecursively(d);
                } catch (IOException exc) {
                	logger.warn("Cannot delete artifact data work dir " + d, exc);
                }
			}
			artifactDataWorkDirs.clear();
        }

		private void setArtifactFileToPointToNonExistentFile() {
	        for (ConfigurationItem pojo : convertedPojos.values()) {
				if (pojo instanceof Artifact) {
					((Artifact) pojo).setFile(NO_ARTIFACT_DATA_LOCATION);
				}
			}
	        convertedPojos.clear();
        }

		public <T extends ConfigurationItem> ConfigurationItemEntity toEntity(final T pojo) {
			ConfigurationItemEntity entity;
			if (pojo instanceof SourceArtifact) {
				entity = new ArtifactEntity(pojo.getType());
			} else {
				entity = new ConfigurationItemEntity(pojo.getType());
			}

            entity.setId(pojo.getId());
			setProperties(pojo, entity);
			if (pojo instanceof SourceArtifact) {
				setData((SourceArtifact) pojo, (ArtifactEntity) entity);
			}
			return entity;
		}

		@SuppressWarnings("unchecked")
		public <F extends ConfigurationItem, G extends ConfigurationItemEntity> List<G> toEntity(final List<F> pojos) {
			List<G> entities = Lists.newArrayList();
			for (F each : pojos) {
				entities.add((G) toEntity(each));
			}
			return entities;
		}

		private <T extends ConfigurationItem> void setProperties(final T item, final ConfigurationItemEntity entity) {
			for (PropertyDescriptor each : DescriptorRegistry.getDescriptor(entity.getType()).getPropertyDescriptors()) {
				putProperty(item, entity, each);
			}
		}

		private <T extends ConfigurationItem> void putProperty(final T item, ConfigurationItemEntity entity, final PropertyDescriptor propertyDescriptor) {
			Object valueForEntity = getValueFromPojo(item, propertyDescriptor);
			if (valueForEntity != null) {
				entity.addValue(propertyDescriptor.getName(), valueForEntity);
			}
		}

		private <T extends ConfigurationItem> Object getValueFromPojo(final T pojo, final PropertyDescriptor propertyDescriptor) {
			Object valueFromPojo = propertyDescriptor.get(pojo);
            return convertValueForEntity(propertyDescriptor, valueFromPojo);
		}

        private Object convertValueForEntity(PropertyDescriptor propertyDescriptor, Object valueFromPojo) {
            Object valueForEntity = null;
            switch (propertyDescriptor.getKind()) {
            case STRING:
	            if (valueFromPojo != null && propertyDescriptor.isPassword()) {
		            valueForEntity = PasswordObfuscator.ensureEncrypted(valueFromPojo.toString());
		            break;
	            } // fall-through to next case below, we want the normal behaviour for non-passwords
            case BOOLEAN:
            case INTEGER:
            case ENUM:
                if (valueFromPojo != null) {
                    valueForEntity = valueFromPojo.toString();
                }
                break;
            case SET_OF_STRING:
                valueForEntity = valueFromPojo;
                break;
            case CI:
                if (valueFromPojo != null) {
                    valueForEntity = ((ConfigurationItem) valueFromPojo).getId();
                }
                break;
            case SET_OF_CI:
                @SuppressWarnings("unchecked") Set<Object> referencedCis = (Set<Object>) valueFromPojo;
                Set<String> result = Sets.newHashSet();
                if (referencedCis != null) {
                    for (Object each : referencedCis) {
                        result.add(((ConfigurationItem) each).getId());
                    }
                }
                valueForEntity = result;
                break;
            case MAP_STRING_STRING:
	            @SuppressWarnings("unchecked") Map<String, String> map = (Map<String, String>) valueFromPojo;
	            if (map != null) {
					// Defensive copy
					return newHashMap(map);
	            } else return null;
            default:
                throw new IllegalStateException("Should not end up here!");
            }
            return valueForEntity;
        }

        private <T extends SourceArtifact> void setData(final T pojo, final ArtifactEntity entity) {
			if (pojo.getFile() == null) {
				return;
			}

			entity.setFilename(pojo.getFile().getName());
			if (pojo instanceof FolderArtifact) {
				setFolderData((FolderArtifact) pojo, entity);
			} else if (pojo instanceof FileArtifact) {
				setFileData((FileArtifact) pojo, entity);
			} else {
				throw new IllegalArgumentException("Unknown subclass of " + SourceArtifact.class + ": " + pojo.getClass());
			}
		}

		private <T extends FolderArtifact> void setFolderData(final T item, final ArtifactEntity entity) {
			try {
				OverthereFile file = item.getFile();
				byte[] implodedFolderData = Imploder.implode(((LocalFile) file).getFile());
				entity.setData(newInputStreamSupplier(implodedFolderData));
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot read artifact folder data from " + item, exc);
			}
		}

		private <T extends FileArtifact> void setFileData(final T item, final ArtifactEntity entity) {
			entity.setData(new InputSupplier<InputStream>() {
                public InputStream getInput() throws IOException {
                    return item.getFile().getInputStream();
                }
			});
		}
	}

	public <T extends ConfigurationItem> ConfigurationItemEntity toEntity(final T pojo) {
		Context context = new Context();
		try {
			return context.toEntity(pojo);
		} finally {
			context.destroy();
		}
	}

	public void setWorkDir(File workDir) {
		this.workDir = workDir;
	}

    private static final Logger logger = LoggerFactory.getLogger(PojoConverter.class);

}
