package com.xebialabs.deployit.jcr;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.jcr.JcrConstants.*;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.ENVIRONMENTS;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.INFRASTRUCTURE;

import java.io.File;
import java.io.IOException;
import java.util.Calendar;
import java.util.Map;

import javax.jcr.NamespaceRegistry;
import javax.jcr.Node;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.nodetype.NodeTypeManager;
import javax.jcr.nodetype.NodeTypeTemplate;
import javax.jcr.security.AccessControlEntry;
import javax.jcr.security.AccessControlManager;
import javax.jcr.security.AccessControlPolicy;
import javax.jcr.security.AccessControlPolicyIterator;
import javax.jcr.security.Privilege;

import org.apache.jackrabbit.api.JackrabbitRepository;
import org.apache.jackrabbit.api.security.JackrabbitAccessControlList;
import org.apache.jackrabbit.core.RepositoryImpl;
import org.apache.jackrabbit.core.config.RepositoryConfig;
import org.apache.jackrabbit.core.security.principal.EveryonePrincipal;
import org.apache.jackrabbit.value.ValueFactoryImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.Resource;
import org.xml.sax.InputSource;

import com.xebialabs.deployit.ReleaseInfo;
import com.xebialabs.deployit.repository.internal.Root;


public class JackrabbitRepositoryFactoryBean implements InitializingBean, FactoryBean<Repository>, DisposableBean {

	private Resource homeDir;

	private Resource configuration;

	private JackrabbitRepository repository;

	private boolean createHomeDirIfNotExists;

	@Override
	public void afterPropertiesSet() throws IOException, RepositoryException {
		InputSource configurationInputSource = new InputSource(configuration.getInputStream());
		File homeDirFile = homeDir.getFile();
		if (!homeDirFile.exists() && !createHomeDirIfNotExists) {
			throw new RepositoryException("Jackrabbit home dir " + homeDirFile + " does not exist");
		}

		RepositoryConfig repositoryConfig = RepositoryConfig.create(configurationInputSource, homeDirFile.getAbsolutePath());
		repository = RepositoryImpl.create(repositoryConfig);
		if (logger.isDebugEnabled())
			logger.debug("Instantiated JackRabbit repository");
	}

	public void configureJcrRepositoryForDeployit() {
		new JcrTemplate(repository).executeAsAdmin(new JcrCallback<Object>() {
			@Override
			public Object doInJcr(final Session session) throws IOException, RepositoryException {
				final NamespaceRegistry namespaceRegistry = session.getWorkspace().getNamespaceRegistry();
				namespaceRegistry.registerNamespace(DEPLOYIT_NAMESPACE_PREFIX, DEPLOYIT_NAMESPACE_URI);

				final NodeTypeManager typeManager = session.getWorkspace().getNodeTypeManager();
				createMixinNodeType(typeManager, CONFIGURATION_ITEM_NODETYPE_NAME);
				createMixinNodeType(typeManager, ARTIFACT_NODETYPE_NAME);

				createRoot(session, APPLICATIONS.getRootNodeName());
				createRoot(session, INFRASTRUCTURE.getRootNodeName());
				createRoot(session, ENVIRONMENTS.getRootNodeName());

				createMixinNodeType(typeManager, TASK_NODETYPE_NAME);
				createMixinNodeType(typeManager, STEP_NODETYPE_NAME);
				createNode(session, TASKS_NODE_NAME);

				createMixinNodeType(typeManager, CONFIGURATION_NODETYPE_NAME);
				createNode(session, CONFIGURATION_NODE_NAME, CONFIGURATION_NODETYPE_NAME);

				createNode(session, SECURITY_NODE_NAME);
				setAccessControlNode(CONFIGURATION_NODE_ID, session, new PrivilegeCallback() {
                    @Override
                    public boolean doWithAcl(JackrabbitAccessControlList acl, AccessControlManager acm) throws RepositoryException {
                        Privilege allPrivilege = acm.privilegeFromName(Privilege.JCR_ALL);
                        Privilege readPrivilege = acm.privilegeFromName(Privilege.JCR_READ);

                        acl.addEntry(EveryonePrincipal.getInstance(), new Privilege[]{allPrivilege}, false);
                        acl.addEntry(EveryonePrincipal.getInstance(), new Privilege[]{readPrivilege}, true);
                        return true;
                    }
                });

				setAccessControlOnRootNode(session);
                setRepoVersion(session);
				return null;
			}

		});
	}

    private void setRepoVersion(Session session) throws RepositoryException {
        Node node = session.getRootNode().addNode(VERSIONS_NODE_NAME);
        node.setProperty("deployit", ReleaseInfo.getReleaseInfo().getVersion());
        session.save();
    }

	private void createMixinNodeType(final NodeTypeManager typeManager, final String nodetypeName) throws RepositoryException {
		final NodeTypeTemplate nodeTypeTemplate = typeManager.createNodeTypeTemplate();
		nodeTypeTemplate.setName(nodetypeName);
		nodeTypeTemplate.setQueryable(true);
		nodeTypeTemplate.setAbstract(false);
		nodeTypeTemplate.setMixin(true);

		typeManager.registerNodeType(nodeTypeTemplate, false);
	}

	private void createRoot(final Session session, final String rootNodeName) throws RepositoryException {
		Node node = session.getRootNode().addNode(rootNodeName);
		node.addMixin(JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME);
		node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, "internal." + Root.class.getSimpleName());
		node.setProperty(ID_PROPERTY_NAME, rootNodeName);
		node.setProperty(LAST_MODIFIED_DATE_PROPERTY_NAME, Calendar.getInstance());

        setAccessControlNode("/" + rootNodeName, session, new PrivilegeCallback() {
            @Override
            public boolean doWithAcl(JackrabbitAccessControlList acl, AccessControlManager acm) throws RepositoryException {
                Privilege read = acm.privilegeFromName(Privilege.JCR_READ);
                final EveryonePrincipal everyone = EveryonePrincipal.getInstance();
                for (AccessControlEntry entry : acl.getAccessControlEntries()) {
                    if (entry.getPrincipal().equals(everyone)) {
                        acl.removeAccessControlEntry(entry);
                    }
                }

                // First do a transitive deny for this node and all its children
                acl.addEntry(everyone, new Privilege[]{read}, false);
                // Then do a non-transitive grant for this node (not propagating to the children)
                Map<String, Value> restrictions = newHashMap();
                restrictions.put("rep:glob", ValueFactoryImpl.getInstance().createValue(""));
                acl.addEntry(everyone, new Privilege[] {read}, true, restrictions);
                return true;
            }
        });
	}

	private void createNode(final Session session, final String nodeName, final String... mixinNodeTypeNames) throws RepositoryException {
		Node rootNode = session.getRootNode();
		Node addedNode = rootNode.addNode(nodeName);
		for (String each : mixinNodeTypeNames) {
			addedNode.addMixin(each);
		}

		session.save();
	}

	protected void setAccessControlNode(final String node, final Session session, PrivilegeCallback callback) throws RepositoryException {
		AccessControlManager acm = session.getAccessControlManager();
		AccessControlPolicyIterator applicablePolicies = acm.getApplicablePolicies(node);

		boolean policySet = false;
		while (!policySet && applicablePolicies.hasNext()) {
			AccessControlPolicy each = applicablePolicies.nextAccessControlPolicy();
			if (!(each instanceof JackrabbitAccessControlList))
				continue;

			JackrabbitAccessControlList acl = (JackrabbitAccessControlList) each;

            policySet = callback.doWithAcl(acl, acm);

			logger.debug("Setting {} on {}", acl, node);
			acm.setPolicy(node, acl);
		}

        if (!policySet) {
            throw new IllegalStateException("Could not set permission on node " + node);
        }
        session.save();

	}

	protected void setAccessControlOnRootNode(final Session session) throws RepositoryException {
		AccessControlManager acm = session.getAccessControlManager();
		AccessControlPolicyIterator applicablePolicies = acm.getApplicablePolicies("/");

		boolean policySet = false;
		while (!policySet && applicablePolicies.hasNext()) {
			AccessControlPolicy each = applicablePolicies.nextAccessControlPolicy();
			if (!(each instanceof JackrabbitAccessControlList))
				continue;

			JackrabbitAccessControlList acl = (JackrabbitAccessControlList) each;

			Privilege readPrivilege = acm.privilegeFromName(Privilege.JCR_READ);

			final EveryonePrincipal everyone = EveryonePrincipal.getInstance();
			for (AccessControlEntry entry : acl.getAccessControlEntries()) {
				if (entry.getPrincipal().equals(everyone)) {
					acl.removeAccessControlEntry(entry);
				}
			}

			acl.addEntry(everyone, new Privilege[]{readPrivilege}, true);

			logger.debug("Setting {} on root", acl);
			acm.setPolicy("/", acl);
		}
	}

	protected void setAccessControlListOnConfigurationNode(final Session session) throws RepositoryException {
		AccessControlManager acm = session.getAccessControlManager();
		AccessControlPolicyIterator applicablePolicies = acm.getApplicablePolicies(CONFIGURATION_NODE_ID);

		boolean policySet = false;
		while (!policySet && applicablePolicies.hasNext()) {
			AccessControlPolicy each = applicablePolicies.nextAccessControlPolicy();
			if (!(each instanceof JackrabbitAccessControlList))
				continue;

			JackrabbitAccessControlList acl = (JackrabbitAccessControlList) each;
			for (AccessControlEntry eachEntry : acl.getAccessControlEntries()) {
				acl.removeAccessControlEntry(eachEntry);
			}

			Privilege allPrivilege = acm.privilegeFromName(Privilege.JCR_ALL);
			Privilege readPrivilege = acm.privilegeFromName(Privilege.JCR_READ);

			acl.addEntry(EveryonePrincipal.getInstance(), new Privilege[]{allPrivilege}, false);
			acl.addEntry(EveryonePrincipal.getInstance(), new Privilege[]{readPrivilege}, true);
			acm.setPolicy(CONFIGURATION_NODE_ID, acl);
			policySet = true;
		}

		if (!policySet) {
			throw new IllegalStateException("Could not set permission on preferences node " + CONFIGURATION_NODE_ID);
		}
		session.save();
	}

	@Override
	public Repository getObject() {
		return repository;
	}

	@Override
	public void destroy() {
		repository.shutdown();
	}

	@Override
	public Class<Repository> getObjectType() {
		return Repository.class;
	}

	@Override
	public boolean isSingleton() {
		return true;
	}

	public Resource getHomeDir() {
		return homeDir;
	}

	@Required
	public void setHomeDir(final Resource homeDir) {
		this.homeDir = homeDir;
	}

	public Resource getConfiguration() {
		return configuration;
	}

	@Required
	public void setConfiguration(final Resource configuration) {
		this.configuration = configuration;
	}

	public boolean isCreateHomeDirIfNotExists() {
		return createHomeDirIfNotExists;
	}

	public void setCreateHomeDirIfNotExists(boolean autoCreateRepositoryDir) {
		this.createHomeDirIfNotExists = autoCreateRepositoryDir;
	}

	private static final Logger logger = LoggerFactory.getLogger(JackrabbitRepositoryFactoryBean.class);


    private interface PrivilegeCallback {
        boolean doWithAcl(JackrabbitAccessControlList acl, AccessControlManager acm) throws RepositoryException;
    }
}
