package com.xebialabs.deployit.task.deployment;

import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.repository.ChangeSet;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.SecurityCallback;
import com.xebialabs.deployit.security.SecurityTemplate;
import com.xebialabs.deployit.security.UsernameAndPasswordCredentials;
import com.xebialabs.deployit.task.DeploymentTask;

import java.util.List;

import static com.google.common.collect.Lists.newArrayList;

@SuppressWarnings("serial")
public class UndeploymentTask extends DeploymentTask {

    private DeltaSpecification specification;
    private DeployedApplication existingDeployment;
    @SuppressWarnings("unused")
    private transient PojoConverter pojoConverter;

    public UndeploymentTask(DeltaSpecification specification, final DeployedApplication existingDeployment, final List<? extends DeploymentStep> steps, final RepositoryService repositoryService, PojoConverter pojoConverter,
                            PojoConverter.Context... pojoConverterContexts) {
		super(steps, repositoryService, pojoConverterContexts);
        this.specification = specification;
        this.existingDeployment = existingDeployment;
        this.pojoConverter = pojoConverter;
        initFromDeployment(existingDeployment);
    }

    @Override
    public void performPreFlightChecks() {
        UsernameAndPasswordCredentials owner = getOwnerCredentials();
		SecurityTemplate.executeAs(owner.getUsername(), owner.getPassword(), new SecurityCallback<Object>() {
			@Override
			public Object doAs() {
                ChangeSet changeset = determineRepositoryChanges();
                repositoryService.checkReferentialIntegrity(changeset);
				return null;
			}
		});
    }

    private ChangeSet determineRepositoryChanges() {
        List<String> roesToDelete = newArrayList();
        // First delete the deployment, otherwise, the mappings cannot be deleted as they're referenced.
        roesToDelete.add(existingDeployment.getId());
        for (Delta delta : specification.getDeltas()) {
            // All delta's should be DESTROY...
            roesToDelete.add(delta.getPrevious().getId());
        }

        ChangeSet batchUpdate = new ChangeSet();
        batchUpdate.setDeleteEntityIds(roesToDelete);
        return batchUpdate;
    }

    @Override
	public void doAfterTaskStateChangedToDone() {
		UsernameAndPasswordCredentials owner = getOwnerCredentials();
		SecurityTemplate.executeAs(owner.getUsername(), owner.getPassword(), new SecurityCallback<Object>() {
			@Override
			public Object doAs() {
                ChangeSet changeset = determineRepositoryChanges();
                repositoryService.execute(changeset);
				return null;
			}
		});
	}

    @Override
    public DeploymentType getDeploymentType() {
        return DeploymentTask.DeploymentType.UNDEPLOY;
    }
}
