package com.xebialabs.deployit.task;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.plugin.api.execution.Step;

import java.util.ArrayList;
import java.util.List;

import static com.google.common.base.Preconditions.*;

@SuppressWarnings("serial")
class StepList extends ArrayList<TaskStep> {

	private volatile int currentStepNr;

    StepList(List<? extends Step> steps) {
       super(new ArrayList<TaskStep>(Collections2.transform(steps, new Function<Step, TaskStep>() {
            @Override
            public TaskStep apply(final Step from) {
                return new TaskStep(from);
            }
        })));
        currentStepNr = 0;
    }

    StepList(List<TaskStep> steps, int currentStepNr) {
        super(steps);
        this.currentStepNr = currentStepNr;
    }

    public int getNrOfSteps() {
        return size();
    }

    public int getCurrentStepNr() {
        return currentStepNr;
    }

    public TaskStep getStep(int stepNr) {
	    return get(stepNumberToIndex(stepNr));
    }

	private int stepNumberToIndex(int stepNr) {
		if (stepNr < 1 || stepNr > getNrOfSteps()) {
		    throw new IllegalArgumentException(stepNr + " is not a valid step number");
		}
		return stepNr - 1;
	}

	public TaskStep getNextStep() {
        return getStep(++currentStepNr);
    }

    public boolean hasMoreSteps() {
        return currentStepNr < getNrOfSteps();
    }

    public void rollback() {
        currentStepNr--;
    }

	public void move(int stepNr, int pos) {
		checkState(currentStepNr == 0, "Cannot move when iteration over StepList has started.");
		int index = stepNumberToIndex(stepNr);
		int newIndex = stepNumberToIndex(pos);
		TaskStep step = get(index);
		remove(index);
		add(newIndex, step);
	}
}
