package com.xebialabs.deployit.service.importer;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.server.api.importer.*;
import com.xebialabs.deployit.service.importer.reader.DeployableConfigurationItemReader;
import com.xebialabs.deployit.service.importer.reader.ManifestReader;
import de.schlichtherle.truezip.file.TFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.service.importer.reader.DeployableConfigurationItemReader.isMiddlewareResource;
import static java.util.Collections.sort;

public class ManifestBasedDarImporter implements ListableImporter {
	public static final String TEMPORARY_FILES = "temporaryFiles";

	@Override
	public List<String> list(File directory) {
		List<String> packages = newArrayList(PackageScanner.scan(directory));
		sort(packages);
		return packages;
	}

	@Override
	public boolean canHandle(ImportSource source) {
		File file = source.getFile();
		return PackageScanner.isDarPackage(file) || PackageScanner.isExplodedPackage(file);
	}

	@Override
	public PackageInfo preparePackage(ImportSource source, ImportingContext ctx) {
		PackageInfo packageInfo = new PackageInfo(source);
		updatePackageInfoWithManifestValues(packageInfo);
		ctx.setAttribute(TEMPORARY_FILES, Lists.<TFile>newArrayList());
		return packageInfo;
	}

	private void updatePackageInfoWithManifestValues(final PackageInfo packageInfo) {
		Manifest manifest = getManifest(packageInfo);
		ManifestReader.verifyPackageFormatVersion(manifest);

		packageInfo.setApplicationName(ManifestReader.getApplicationName(manifest));
		packageInfo.setApplicationVersion(ManifestReader.getApplicationVersion(manifest));
	}

	@Override
	public ImportedPackage importEntities(PackageInfo packageInfo, ImportingContext ctx) {
		ImportedPackage importedPackage = new ImportedPackage(packageInfo);

		Manifest manifest = getManifest(packageInfo);

		createDeployableEntities(packageInfo, manifest.getEntries(), importedPackage, ctx);

		return importedPackage;
	}

	private void createDeployableEntities(PackageInfo packageInfo, Map<String, Attributes> entries, ImportedPackage importedPackage, ImportingContext ctx) {
		for (Map.Entry<String, Attributes> entry : entries.entrySet()) {
			if (isMiddlewareResource(entry)) {
				importedPackage.addDeployable(DeployableConfigurationItemReader.readMiddlewareConfiguration(importedPackage.getDeploymentPackage(), entry));
			} else {
				importedPackage.addDeployable(DeployableConfigurationItemReader.readArtifact(importedPackage.getDeploymentPackage(), packageInfo.getSource(), entry, ctx));
			}
		}
	}

	private Manifest getManifest(PackageInfo packageInfo) {
		TFile manifestFileForUnpackedPackage = PackageScanner.getManifestFileFromImportSource(packageInfo.getSource());
		return ManifestReader.readManifest(manifestFileForUnpackedPackage);
	}

	@Override
	public void cleanUp(PackageInfo packageInfo, ImportingContext ctx) {
		packageInfo.getSource().cleanUp();
		List<TFile> files = ctx.getAttribute(TEMPORARY_FILES);
		for (TFile file : files) {
			logger.debug("Cleaning up temporary file {}", file);
			try {
				if (file.isDirectory()) {
					file.rm_r();
				} else {
					file.rm();
				}
			} catch (IOException e) {
				logger.error("Couldn't clean up file {}", file, e);
			}
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(ManifestBasedDarImporter.class);
}
