package com.xebialabs.deployit.service.comparison;

import com.google.common.base.Joiner;
import com.google.common.collect.*;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import org.springframework.stereotype.Component;

import java.io.StringWriter;
import java.util.*;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;

@Component
public class Comparator {

    private static final String PASSWORD_OBFUSCATION = "********";

    public ListMultimap<String, String> compare(RepositoryObjectEntity reference, RepositoryObjectEntity... entities) {
	    ComputingLinkedMap comparison = new ComputingLinkedMap(entities.length + 1);

	    checkCiTypes(reference.getType(), entities);

	    Descriptor descriptor = DescriptorRegistry.getDescriptor(reference.getType());
	    int nrCi = 0;

	    List<RepositoryObjectEntity> entitiesToCompare = Lists.newArrayList();
	    entitiesToCompare.add(reference);
	    entitiesToCompare.addAll(Arrays.asList(entities));
	    for (RepositoryObjectEntity compareEntity : entitiesToCompare) {
	      writeFields(comparison, compareEntity, nrCi);
	      writeProperties(comparison, compareEntity, descriptor, nrCi);
	      nrCi++;
	    }

	    LinkedListMultimap<String, String> retval = LinkedListMultimap.create();
	    for (String k : comparison.keySet()) {
		    retval.putAll(k, comparison.get(k));
	    }
	    return retval;
    }

    @SuppressWarnings("unchecked")
    private void writeProperties(final ComputingLinkedMap comparison, final RepositoryObjectEntity entity, final Descriptor descriptor, final int nrCi) {
        final Collection<PropertyDescriptor> descriptors = descriptor.getPropertyDescriptors();
        for (PropertyDescriptor propertyDescriptor : descriptors) {
            final String key = propertyDescriptor.getName();
            Object value = entity.getValue(key);
            if (propertyDescriptor.isPassword()) {
                value = PASSWORD_OBFUSCATION;
            }
            switch (propertyDescriptor.getKind()) {
                case BOOLEAN:
                case INTEGER:
                case STRING:
                case ENUM:
                case CI:
                    comparison.put(key, (String) value, nrCi);
                    break;
                case SET_OF_STRING:
                case SET_OF_CI:
                    handleSetOfString(comparison, key, (Collection<String>) value, nrCi);
                    break;
                case MAP_STRING_STRING:
	                handleMap(comparison, key, (Map<String, String>) value, nrCi);
	                break;
	            default:
	                throw new IllegalStateException("Should not end up here!");
            }
        }
    }

	private void handleMap(ComputingLinkedMap comparison, String key, Map<String, String> value, int nrCi) {
		if (value != null) {
			// Order the keys in the Map
			TreeSet<String> set = new TreeSet(value.keySet());
			for (String k : set) {
				comparison.put(key + ": " + k, value.get(k), nrCi);
			}
		}
	}

	private void handleSetOfString(final ComputingLinkedMap comparison, final String key, final Collection<String> value, final int nrCi) {
        if (value != null) {
            final ArrayList<String> list = new ArrayList<String>(value);
            Collections.sort(list);
            comparison.put(key, Joiner.on(",").join(list), nrCi);
        }
    }

    private void writeFields(final ComputingLinkedMap comparison, final RepositoryObjectEntity entity, int nrCi) {
        comparison.put("id", entity.getId(), nrCi);
        comparison.put("type", entity.getType().toString(), nrCi);
    }

    private void checkCiTypes(final Type referenceType, final RepositoryObjectEntity... entities) {
        for (RepositoryObjectEntity entity : entities) {
            checkArgument(entity.getType().equals(referenceType), "Not all configuration items are of type %s", referenceType);
        }
    }

	static class ComputingLinkedMap extends LinkedHashMap<String, List<String>> {
		private final int listSize;
		private String[] strings;

		public ComputingLinkedMap(int listSize) {
			super();
			this.listSize = listSize;
			strings = new String[listSize];
			Arrays.fill(strings, "");
		}

		@Override
		public List<String> get(Object o) {
			List<String> value = super.get(o);
			if (value == null) {
				value = Lists.newArrayList(strings);
				super.put((String) o, value);
			}

			return value;
		}

		public void put(String key, String value, int position) {
			List<String> values = get(key);
			values.remove(position);
			values.add(position, value);
		}
	}
}
