package com.xebialabs.deployit.security.permission;

import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Iterables.toArray;

import java.util.Collection;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Predicate;
import com.xebialabs.deployit.security.AccessControlService;
import com.xebialabs.deployit.security.JcrAccessControlEntry;
import com.xebialabs.deployit.security.SecurityServiceLocator;

public class PrivilegesHelper {

	static void grantPrivileges(List<String> privileges, final String principal, String node, boolean transitive) {
		final AccessControlService accessControlService = SecurityServiceLocator.getAccessControlService();
		final Collection<JcrAccessControlEntry> acls = accessControlService.getAccessControlEntries(node);
		JcrAccessControlEntry found = findAclForPrincipal(principal, acls, true, transitive);

		if (found == null) {
			found = new JcrAccessControlEntry(principal, true, privileges, transitive);
			acls.add(found);
		} else {
			found.getPrivileges().addAll(privileges);
		}

		accessControlService.setAccessControlEntries(node, toArray(acls, JcrAccessControlEntry.class));
	}

	static void denyPrivileges(List<String> privileges, final String principal, String node, boolean transitive) {
		final AccessControlService accessControlService = SecurityServiceLocator.getAccessControlService();
		final Collection<JcrAccessControlEntry> acls = accessControlService.getAccessControlEntries(node);
		JcrAccessControlEntry found = findAclForPrincipal(principal, acls, false, transitive);

		if (found == null) {
			found = new JcrAccessControlEntry(principal, false, privileges, transitive);
			acls.add(found);
		} else {
			found.getPrivileges().addAll(privileges);
		}

		accessControlService.setAccessControlEntries(node, toArray(acls, JcrAccessControlEntry.class));
	}

	static void revokePrivileges(List<String> privileges, final String principal, String node, boolean transitive) {
		final AccessControlService accessControlService = SecurityServiceLocator.getAccessControlService();
		final Collection<JcrAccessControlEntry> acls = accessControlService.getAccessControlEntries(node);
		Collection<JcrAccessControlEntry> found = findAclsForPrincipal(principal, acls);

		for (JcrAccessControlEntry entry : found) {
			if (entry.isTransitive() == transitive) {
				entry.getPrivileges().removeAll(privileges);
			}
		}

		accessControlService.setAccessControlEntries(node, toArray(acls, JcrAccessControlEntry.class));

	}

	static JcrAccessControlEntry findAclForPrincipal(final String principal, Collection<JcrAccessControlEntry> acls, boolean granted, boolean transitive) {
		final Collection<JcrAccessControlEntry> aclsForPrincipal = findAclsForPrincipal(principal, acls);

		JcrAccessControlEntry found = null;
		for (JcrAccessControlEntry jcrAccessControlEntry : aclsForPrincipal) {
			if (jcrAccessControlEntry.isAllow() == granted && jcrAccessControlEntry.isTransitive() == transitive) {
				found = jcrAccessControlEntry;
				break;
			}
		}
		return found;
	}

	static Collection<JcrAccessControlEntry> findAclsForPrincipal(final String principal, Collection<JcrAccessControlEntry> acls) {
		final Collection<JcrAccessControlEntry> aclsForPrincipal = filter(acls, new Predicate<JcrAccessControlEntry>() {
			@Override
			public boolean apply(JcrAccessControlEntry input) {
				return input.getPrincipalName().equals(principal);
			}
		});

		logger.debug("Found [{}] acls for principal {}", aclsForPrincipal, principal);
		return aclsForPrincipal;
	}

	private static final Logger logger = LoggerFactory.getLogger(PermissionHandler.class);

}
