package com.xebialabs.deployit.repository;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.jcr.JcrConstants.ARTIFACT_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CREATING_TASK_ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.DATA_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.FILENAME_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_DATE_PROPERTY_NAME;
import static com.xebialabs.deployit.repository.JcrPathHelper.getIdFromAbsolutePath;
import static javax.jcr.nodetype.NodeType.NT_FROZEN_NODE;
import static javax.jcr.query.Query.JCR_SQL2;
import static org.apache.jackrabbit.JcrConstants.JCR_FROZENMIXINTYPES;
import static org.apache.jackrabbit.JcrConstants.JCR_FROZENUUID;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import javax.jcr.nodetype.NodeType;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;

import org.apache.jackrabbit.spi.commons.value.QValueValue;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Sets;
import com.google.common.io.InputSupplier;
import com.xebialabs.deployit.jcr.RuntimeRepositoryException;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;

/**
 * Reads an {@link RepositoryObjectEntity} from a JCR {@link Node}.
 */
class EntityNodeReader {

	private Session session;

	private Node node;

	private boolean readFully;

	public EntityNodeReader(final Session session, final Node node, final boolean readFully) {
		this.session = session;
		this.node = node;
		this.readFully = readFully;
	}

	@SuppressWarnings("unchecked")
	public <T extends RepositoryObjectEntity> T getEntity() throws RepositoryException, IOException {
		RepositoryObjectEntity entity = instantiateEntity();
		copyEntityMetadata(entity);
		copyData(entity);
		copyValuesIntoEntity(entity);
		return (T) entity;
	}

	private void copyEntityMetadata(final RepositoryObjectEntity entity) throws RepositoryException {
		entity.setId(node.getProperty(ID_PROPERTY_NAME).getString());
		entity.setLastModified(node.getProperty(LAST_MODIFIED_DATE_PROPERTY_NAME).getDate());
		if (node.hasProperty(CREATING_TASK_ID_PROPERTY_NAME)) {
			entity.setCreatingTaskId(node.getProperty(CREATING_TASK_ID_PROPERTY_NAME).getString());
		}
	}

	private void copyData(final RepositoryObjectEntity entity) throws RepositoryException, IOException {
		if (entity instanceof ArtifactEntity) {
			ArtifactEntity artifact = (ArtifactEntity) entity;
			((ArtifactEntity) entity).setFilename(node.getProperty(FILENAME_PROPERTY_NAME).getString());
			if (readFully) {
				artifact.setData(new InputSupplier<InputStream>() {
                    public InputStream getInput() throws IOException {
	                    try {
	                        return node.getProperty(DATA_PROPERTY_NAME).getBinary().getStream();
                        } catch (RepositoryException exc) {
                        	throw new IOException("Cannot read artifact data from " + entity.getId(), exc);
                        }
                    }
				});
			}
		}

	}

	private void copyValuesIntoEntity(final RepositoryObjectEntity entity) throws RepositoryException {
		Descriptor descriptor = DescriptorRegistry.getDescriptor(entity.getType());
		if (descriptor == null) {
			@SuppressWarnings("unused")
			Descriptor d2 = DescriptorRegistry.getDescriptor(entity.getType());
			throw new RuntimeException("Cannot find descriptor for " + entity.getType());
		}
		for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
			if (!node.hasProperty(pd.getName()) && !pd.isAsContainment()) {
				continue;
			}

			switch (pd.getKind()) {
			case BOOLEAN:
			case INTEGER:
			case STRING:
			case ENUM:
				copyPrimitivePropertyFromNode(entity, pd);
				break;
			case SET_OF_STRING:
				copySetOfStringsPropertyFromNode(entity, pd);
				break;
			case CI:
				copyConfigurationItemPropertyFromNode(entity, pd);
				break;
			case SET_OF_CI:
				copySetOfConfigurationItemsPropertyFromNode(entity, pd);
				break;
			case MAP_STRING_STRING:
				copyMapPropertyFromNode(entity, pd);
				break;
			default:
				throw new IllegalArgumentException("Cannot convert property " + pd.getName() + " because it is of unsupported kind " + pd.getKind());
			}
		}
	}

	private RepositoryObjectEntity instantiateEntity() throws RepositoryException, IOException {
		final Type type = Type.valueOf(node.getProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString());
		final Collection<String> mixinNodeTypeNames = getMixinNodeTypeNames();

		// check whether this entity is an artifact first, because an artifact
		// is also a configuration-item
		if (mixinNodeTypeNames.contains(ARTIFACT_NODETYPE_NAME)) {
			return new ArtifactEntity(type);
		}

		if (mixinNodeTypeNames.contains(CONFIGURATION_ITEM_NODETYPE_NAME)) {
			return new ConfigurationItemEntity(type);
		}

		throw new IllegalArgumentException("Cannot determine whether JCR node " + JcrPathHelper.getIdFromAbsolutePath(node.getPath())
		        + " is a configuration item");
	}

	protected Collection<String> getMixinNodeTypeNames() throws RepositoryException {
		if (nodeIsFrozen()) {
			return getMixinNodeTypeNamesFromFrozenNode();
		} else {
			return getMixinNodeTypeNamesFromRegularNode();
		}
	}

	protected Collection<String> getMixinNodeTypeNamesFromFrozenNode() throws RepositoryException {
		final Property property = node.getProperty(JCR_FROZENMIXINTYPES);
		final ArrayList<Value> frozenMixinTypes = newArrayList(property.getValues());
		return Collections2.transform(frozenMixinTypes, new Function<Value, String>() {
			@Override
			public String apply(Value from) {
				try {
					return from.getString();
				} catch (RepositoryException e) {
					throw new RuntimeRepositoryException("Cannot convert Value into String", e);
				}
			}
		});
	}

	protected Collection<String> getMixinNodeTypeNamesFromRegularNode() throws RepositoryException {
		final ArrayList<NodeType> mixinNodeTypes = newArrayList(node.getMixinNodeTypes());
		return Collections2.transform(mixinNodeTypes, new Function<NodeType, String>() {
			@Override
			public String apply(NodeType from) {
				return from.getName();
			}
		});
	}

private void copyMapPropertyFromNode(RepositoryObjectEntity entity, PropertyDescriptor pd) throws RepositoryException {
		Map<String, String> map = newHashMap();
		for (Value value : node.getProperty(pd.getName()).getValues()) {
			String v = value.getString();
			int i = v.indexOf("=");
			map.put(v.substring(0, i), v.substring(i + 1));
		}
		entity.addValue(pd.getName(), map);
	}

	private void copyPrimitivePropertyFromNode(RepositoryObjectEntity entity, PropertyDescriptor pd) throws RepositoryException {
		entity.addValue(pd.getName(), node.getProperty(pd.getName()).getString());
	}

	private void copySetOfStringsPropertyFromNode(RepositoryObjectEntity entity, PropertyDescriptor pd) throws RepositoryException {
		Set<String> setOfStrings = newHashSet();
		for (Value each : node.getProperty(pd.getName()).getValues()) {
			setOfStrings.add(each.getString());
		}
		entity.addValue(pd.getName(), setOfStrings);
	}

	private void copyConfigurationItemPropertyFromNode(RepositoryObjectEntity entity, PropertyDescriptor pd) throws RepositoryException {
		if (pd.isAsContainment()) {
			entity.addValue(pd.getName(), getIdFromAbsolutePath(node.getParent().getPath()));
		} else {
			Value value = node.getProperty(pd.getName()).getValue();
			entity.addValue(pd.getName(), getReferencedCiId(value));
		}
	}

	private void copySetOfConfigurationItemsPropertyFromNode(RepositoryObjectEntity entity, PropertyDescriptor pd) throws RepositoryException {
		Set<String> referencedCiIds = Sets.newHashSet();
		if (pd.isAsContainment()) {
			SearchParameters params = new SearchParameters().setParent(entity.getId()).setType(pd.getReferencedType());
			SearchQueryBuilder builder = new SearchQueryBuilder(params);
			final Query query = builder.build(session);
			final QueryResult queryResult = query.execute();
			NodeIterator iterator = queryResult.getNodes();
			while (iterator.hasNext()) {
				referencedCiIds.add(getIdFromAbsolutePath(iterator.nextNode().getPath()));
			}
		} else {
			for (Value each : node.getProperty(pd.getName()).getValues()) {
				referencedCiIds.add(getReferencedCiId(each));
			}
		}
		entity.addValue(pd.getName(), referencedCiIds);
	}

	private String getReferencedCiId(Value value) throws RepositoryException {
		String referencedCiUuid = ((QValueValue) value).getQValue().getString();
		Node referencedCi = getNodeByUuid(referencedCiUuid);
		return getIdFromAbsolutePath(referencedCi.getPath());
	}

	protected Node getNodeByUuid(String referencedCiUuid) throws RepositoryException {
		try {
			return session.getNodeByIdentifier(referencedCiUuid);
		} catch (ItemNotFoundException exc) {
			if (nodeIsFrozen()) {
				Node frozenNode = getFrozenNodeByUuid(referencedCiUuid);
				if (frozenNode != null) {
					return frozenNode;
				}
			}
			throw exc;
		}
	}

	private Node getFrozenNodeByUuid(String referencedCiUuid) throws RepositoryException {
		QueryManager queryManager = session.getWorkspace().getQueryManager();
		ValueFactory valueFactory = session.getValueFactory();

		Query query = queryManager.createQuery("SELECT * FROM [" + NT_FROZEN_NODE + "] WHERE [" + JCR_FROZENUUID + "] = $uuid", JCR_SQL2);
		query.bindValue("uuid", valueFactory.createValue(referencedCiUuid));
		QueryResult result = query.execute();
		NodeIterator resultNodes = result.getNodes();
		while (resultNodes.hasNext()) {
			// Assuming that a query by UUID only return one node
			return resultNodes.nextNode();
		}
		return null;
	}

	private boolean nodeIsFrozen() throws RepositoryException {
		return node.isNodeType(NodeType.NT_FROZEN_NODE);
	}

}
