package com.xebialabs.deployit.task;

import static com.xebialabs.deployit.task.TaskStepInfo.StepState.DONE;
import static com.xebialabs.deployit.task.TaskStepInfo.StepState.EXECUTING;
import static com.xebialabs.deployit.task.TaskStepInfo.StepState.FAILED;
import static com.xebialabs.deployit.task.TaskStepInfo.StepState.PENDING;
import static com.xebialabs.deployit.task.TaskStepInfo.StepState.SKIPPED;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Calendar;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.Step;

@SuppressWarnings("serial")
public class TaskStep extends TaskStepInfo {

	private static final String MDC_KEY_STEP_DESCRIPTION = "stepDescription";

	private final Step implementation;

    public TaskStep(Step step) {
        this(step,PENDING);
	}

    TaskStep(Step step, StepState state) {
        super(step.getDescription());
        this.implementation = step;
		setState(state);
    }

    TaskStep(String description, StepState state) {
        super(description);
        setState(state);
        implementation = null;
    }

    public void execute(ExecutionContextAttributes taskContext) {
    	MDC.put(MDC_KEY_STEP_DESCRIPTION, implementation.getDescription());
    	try {
			if ((getState() != PENDING && getState() != FAILED)) {
	            logger.debug("Will not execute: {} with description: {} because it has state: {}", new Object[] { implementation, implementation.getDescription(),getState() });
				return;
			}
	
			setStartDate();
			setState(EXECUTING);
			clearLog();
			logger.info("Executing: {} with description: {}", implementation, implementation.getDescription());
			final TaskStepDeploymentExecutionContext context = new TaskStepDeploymentExecutionContext(taskContext);
			try {
				final Step.Result result = implementation.execute(context);
				setState(result == Step.Result.Success ? DONE : FAILED);
			} catch (Exception exc) {
				context.logError("Step failed", exc);
				setState(FAILED);
			} catch (Throwable t) {
				// Any non-exception throwable is so severe we abort the task!
				context.logError("Step failed badly, aborting!", t);
				Thread.currentThread().interrupt();
				setState(FAILED);
			}
			setCompletionDate();
    	} finally {
    		MDC.remove(MDC_KEY_STEP_DESCRIPTION);
    	}
	}

    protected void transitionExecutingStateToFailedState() {
        if (getState() == StepState.EXECUTING) {
		    setState(StepState.FAILED);
		}
    }

	public Step getImplementation() {
		return implementation;
	}

	protected void setState(StepState state) {
        super.setState(state);
        if(state == FAILED) {
            setFailureCount(getFailureCount() + 1);
        }
		setLastModificationDate();
	}

	protected void setStartDate() {
		super.setStartDate(Calendar.getInstance());
		setLastModificationDate();
	}

	protected void setCompletionDate() {
		super.setCompletionDate(Calendar.getInstance());
		setLastModificationDate();
	}

	public void clearLog() {
		setLog("");
	}

	protected void setLastModificationDate() {
		super.setLastModificationDate(Calendar.getInstance());
	}

	public void skip() {
		setState(SKIPPED);
	}

	public void unskip() {
		setState(PENDING);
	}

    public boolean isSkipped() {
        return getState() == SKIPPED;
    }

    public boolean canSkip() {
        return getState() == PENDING || getState() == FAILED;
    }

	class TaskStepDeploymentExecutionContext implements DeploymentExecutionContext {

		private ExecutionContextAttributes taskContext;

		private Logger stepLogger;

		TaskStepDeploymentExecutionContext(ExecutionContextAttributes taskContext) {
			this.taskContext = taskContext;
			this.stepLogger = LoggerFactory.getLogger(implementation.getClass());
		}

		@Override
		public void logOutput(String output) {
			stepLogger.info(output);
			setLog(getLog() + output + "\n");
			setLastModificationDate();
		}

		@Override
		public void logError(String error) {
			stepLogger.error(error);
			setLog(getLog() + error + "\n");
			setLastModificationDate();
		}

		@Override
		public void logError(String error, Throwable t) {
			stepLogger.error(error, t);
			setLog(getLog() + error + "\n");
			StringWriter stringWriter = new StringWriter();
			t.printStackTrace(new PrintWriter(stringWriter));
			setLog(getLog() + stringWriter.toString() + "\n");
			setLastModificationDate();
		}

		@Override
		public Object getAttribute(String name) {
			Object object = taskContext.getAttribute(name);
			stepLogger.debug("Getting value of attribute {}: {}", name, object);
			return object;
		}

		@Override
		public void setAttribute(String name, Object object) {
			stepLogger.debug("Setting value of attribute {}: {}", name, object);
			taskContext.setAttribute(name, object);
		}

	}

	public boolean isFailed() {
		return getState() == FAILED;
	}

    private static final Logger logger = LoggerFactory.getLogger(TaskStep.class);
}
