package com.xebialabs.deployit.service.validation;

import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.jcr.grouping.Function;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;

import static com.google.common.collect.Lists.newArrayList;

@Component
public class Validator {

    private RepositoryService repositoryService;

	@Autowired
	public Validator(RepositoryService repositoryService) {
		this.repositoryService = repositoryService;
	}

	public Validations validate(RepositoryObjectEntity roe, RepositoryObjectEntity... cisInContext) {
	    return validate(roe, newArrayList(cisInContext));
    }

    public Validations validate(RepositoryObjectEntity roe, List<? extends RepositoryObjectEntity> cisInContext) {
        Validations validations = new Validations();
        ConfigurationItemEntity ci = (ConfigurationItemEntity) roe;
        Type type = ci.getType();
        Descriptor descriptor = DescriptorRegistry.getDescriptor(type);
        for (PropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {

            if (propertyDescriptor.isRequired() && !propertyDescriptor.isAsContainment()) {
                validations.checkRequired(ci, propertyDescriptor);
            }
        }

        for (String key : ci.getValues().keySet()) {
            if (!isPlaceholderField(key) && validations.checkFieldExists(ci, descriptor, key)) {
                PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(key);
                final Object value = ci.getValue(propertyDescriptor);
	            validations.checkType(ci, propertyDescriptor, value, repositoryService, (List<RepositoryObjectEntity>) cisInContext);
            }
        }

        // FIXME: Placeholders are still on the TODO list
        if (descriptor.getPropertyDescriptor("placeholders") != null) {
            final Object placeholdersObject = ci.getValue("placeholders");
	        if (!validations.checkHasSource(ci)) {
				return validations;
	        }
            if (placeholdersObject == null || (placeholdersObject instanceof Collection && ((Collection) placeholdersObject).isEmpty())) {
                String sourceId = (String) ci.getValue("deployable");
                final RepositoryObjectEntity source = repositoryService.read(sourceId);
                final Object placeholdersFromSource = source.getValue("placeholders");
                if (placeholdersFromSource != null && !((Collection<String>) placeholdersFromSource).isEmpty()) {
                    validations.add(new ValidationMessage(ci, "placeholders", "Missing all placeholders from source %s (%s)", sourceId, placeholdersFromSource));
                }
            } else if (validations.checkPlaceholdersOfCorrectType(ci, descriptor, placeholdersObject)) {
                validations.checkPlaceholdersAllFilled(ci, descriptor, placeholdersObject);
                validations.checkPlaceholdersAllPresent(ci, descriptor, placeholdersObject, repositoryService);
            }
        }
        return validations;
    }


    private boolean isPlaceholderField(final String key) {
        return key.equals("placeholders");
    }

    public static class Validations extends ArrayList<ValidationMessage> {

        private void checkRequired(final RepositoryObjectEntity ci, final PropertyDescriptor propertyDescriptor) {
            final Object value = ci.getValue(propertyDescriptor);
            if (value == null || value.toString().trim().isEmpty()) {
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s is required", propertyDescriptor.getName()));
            }
        }

        public boolean hasMessages() {
            return !isEmpty();
        }

        private void checkInteger(final RepositoryObjectEntity ci, final Object value, final PropertyDescriptor propertyDescriptor) {
            if (value != null) {
                try {
                    Integer.parseInt(value.toString());
                } catch (NumberFormatException nfe) {
                    add(new ValidationMessage(ci, propertyDescriptor, "Could not convert %s with value [%s] into an int", propertyDescriptor.getName(), value));
                }
            }
        }

        private void checkBoolean(final RepositoryObjectEntity ci, final Object value, final PropertyDescriptor propertyDescriptor) {
            if (value != null) {
                if ("true".equalsIgnoreCase(value.toString()) || "false".equalsIgnoreCase(value.toString())) {
                    return;
                }
                add(new ValidationMessage(ci, propertyDescriptor, "Could not convert %s with value [%s] into a boolean", propertyDescriptor.getName(), value));
            }
        }

        private void checkEnum(final RepositoryObjectEntity ci, final Object value, final PropertyDescriptor propertyDescriptor) {
            if (value != null) {
                for (String enumValue : propertyDescriptor.getEnumValues()) {
                    if (enumValue.equals(value)) {
                        return;
                    }
                }
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s with value [%s] does not contain correct enum value", propertyDescriptor.getName(), value));
            }
        }

        private boolean checkFieldExists(final RepositoryObjectEntity ci, final Descriptor descriptor, final String key) {
            PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(key);
            if (propertyDescriptor == null || !propertyDescriptor.getName().equals(key)) {
                add(new ValidationMessage(ci, key, "Property %s does not exist on configuration item", key));
                return false;
            }

            return true;
        }

        private void checkCi(final RepositoryObjectEntity ci, final Object value, final PropertyDescriptor propertyDescriptor, final RepositoryService repositoryService, List<RepositoryObjectEntity> cisInContext) {
            if (value == null || propertyDescriptor.isAsContainment()) return;
            if (!ciExists(value.toString(), repositoryService, cisInContext)) {
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s with value [%s] does not point to an existing configuration item", propertyDescriptor.getName(), value));
            }
        }

        private boolean ciExists(String ciId, RepositoryService repositoryService, List<RepositoryObjectEntity> cisInContext) {
            return repositoryService.checkNodeExists(ciId) || inCisContext(ciId, cisInContext);

        }

        private boolean inCisContext(String ciId, List<RepositoryObjectEntity> cisInContext) {
            for (RepositoryObjectEntity entity : cisInContext) {
                if (entity.getId().equals(ciId)) {
                    return true;
                }
            }
            return false;
        }

        private void checkSetOfStrings(final RepositoryObjectEntity ci, final Object value, final PropertyDescriptor propertyDescriptor) {
            if (value == null) return;

            boolean valid = true;
            if (!(value instanceof Collection)) {
                valid = false;
            } else {
                for (Object o : (Collection) value) {
                    valid = valid && o instanceof String;
                }
            }

            if (!valid) {
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s with value %s[%s] does not contain a set of strings", propertyDescriptor.getName(), value.getClass().getSimpleName(), value));
            }
        }

        private void checkSetOfCis(final RepositoryObjectEntity ci, final Object value, final PropertyDescriptor propertyDescriptor, final RepositoryService repositoryService, List<RepositoryObjectEntity> cisInContext) {
            if (value == null || propertyDescriptor.isAsContainment()) return;

            boolean valid = true;
            // Checks are made on Collection, as RestEASY does not always give us Sets but also Lists.
            if (!(value instanceof Collection)) {
                valid = false;
            } else {
                for (Object o : (Collection) value) {
                    valid = valid && o instanceof String && ciExists((String) o, repositoryService, cisInContext);
                }
            }

            if (!valid) {
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s with value %s[%s] does not contain a set of configuration items", propertyDescriptor.getName(), value.getClass().getSimpleName(), value));
            }
        }

        private void checkType(final RepositoryObjectEntity ci, final PropertyDescriptor propertyDescriptor, final Object value, final RepositoryService repositoryService, List<RepositoryObjectEntity> cisInContext) {
            switch (propertyDescriptor.getKind()) {
                case BOOLEAN:
                    checkBoolean(ci, value, propertyDescriptor);
                    break;
                case INTEGER:
                    checkInteger(ci, value, propertyDescriptor);
                    break;
                case STRING:
                    // no-op ;-)
                    break;
                case ENUM:
                    checkEnum(ci, value, propertyDescriptor);
                    break;
                case SET_OF_STRING:
                    checkSetOfStrings(ci, value, propertyDescriptor);
                    break;
                case CI:
                    checkCi(ci, value, propertyDescriptor, repositoryService, cisInContext);
                    break;
                case SET_OF_CI:
                    checkSetOfCis(ci, value, propertyDescriptor, repositoryService, cisInContext);
                    break;
                case MAP_STRING_STRING:
	                checkMapOfString(ci, value, propertyDescriptor, repositoryService);
	                break;
	            default:
	                throw new IllegalStateException("Should not end up here!");
            }
        }

	    private void checkMapOfString(RepositoryObjectEntity ci, Object value, PropertyDescriptor propertyDescriptor, RepositoryService repositoryService) {
		    if (value == null || propertyDescriptor.isAsContainment()) return;

		    boolean valid = true;
		    // Checks are made on Collection, as RestEASY does not always give us Sets but also Lists.
		    if (!(value instanceof Map)) {
		        valid = false;
		    }

		    if (!valid) {
		        add(new ValidationMessage(ci, propertyDescriptor, "Property %s with value %s[%s] does not contain a Map<String, String>", propertyDescriptor.getName(), value.getClass().getSimpleName(), value));
		    }
	    }

        private void checkPlaceholdersAllFilled(final ConfigurationItemEntity ci, final Descriptor descriptor, final Object placeholdersObject) {
            Map<String, String> placeholders = (Map<String, String>) placeholdersObject;
	        for (Map.Entry<String, String> entry : placeholders.entrySet()) {
                if (Strings.nullToEmpty(entry.getValue()).trim().isEmpty()) {
	                add(new ValidationMessage(ci, "placeholders", "Placeholder [%s] requires a value", entry.getKey()));
                }
	        }
        }

        public void checkPlaceholdersAllPresent(final ConfigurationItemEntity ci, final Descriptor descriptor, final Object placeholdersObject, final RepositoryService repositoryService) {
            Map<String, String> placeholders = (Map<String, String>) placeholdersObject;
            final String sourceId = (String) ci.getValue("deployable");
            final RepositoryObjectEntity source = repositoryService.read(sourceId);
            final Set<String> placeholderSet = (Set<String>) source.getValue("placeholders");

	        // DEPLOYITPB-1430
	        if (placeholderSet == null) {
		        return;
	        }

            final Collection<String> placeholderKeys = placeholders.keySet();

            for (String placeholder : placeholderSet) {
                if (!placeholderKeys.contains(placeholder)) {
                    add(new ValidationMessage(ci, "placeholders", "Placeholder [%s] is missing", placeholder));
                }
            }

            for (String placeholder : placeholders.keySet()) {
                if (!placeholderSet.contains(placeholder)) {
                    add(new ValidationMessage(ci, "placeholders", "Extra placeholder [%s] found which is not present in the source", placeholder));
                }
            }
        }

        private boolean checkPlaceholdersOfCorrectType(final ConfigurationItemEntity ci, final Descriptor descriptor, final Object placeholdersObject) {
            if (!(placeholdersObject instanceof Map)) {
                add(new ValidationMessage(ci, "placeholders", "Placeholders in field [placeholders] are not of correct type [%s]", placeholdersObject.getClass().getName()));
                return false;
            }

	        if (Maps.filterEntries((Map<Object,Object>) placeholdersObject, new Predicate<Map.Entry<Object, Object>>() {
		        @Override
		        public boolean apply(Map.Entry<Object, Object> input) {
			        return !(input.getKey() instanceof String && input.getValue() instanceof String);
		        }
	        }).size() > 0) {
		        add(new ValidationMessage(ci, "placeholders", "Placeholders in field [placeholders] are not of correct type."));
                return false;
            }
            return true;
        }

	    public boolean checkHasSource(ConfigurationItemEntity ci) {
		    return ci.getValue("deployable") != null;
	    }
    }


}
