package com.xebialabs.deployit.service.replacement;

import static com.google.common.collect.Maps.newHashMap;

import java.io.Reader;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;

import com.google.common.collect.Maps;
import com.samskivert.mustache.Mustache;
import com.samskivert.mustache.MustacheException;
import com.xebialabs.deployit.service.deployment.DictionaryValueException;

public class MustachePlaceholderReplacer {

	public static final String IGNORE_PLACEHOLDER = "<ignore>";
	public static final String EMPTY_PLACEHOLDER = "<empty>";
	private final Map<String, Object> values;

	public MustachePlaceholderReplacer(Map<String, ? extends Object> values) {
		this.values = new ReplacerMap(values);
	}

	public String replace(String replaceable) throws DictionaryValueException {
		try {
			return Mustache.compiler().compile(replaceable).execute(values);
		} catch (MustacheException me) {
			throw new DictionaryValueException(me, "Could not replace dictionary keys in %s", replaceable);
		}
	}

	public void replace(Reader in, Writer out) {
		Mustache.compiler().compile(in).execute(values, out);
	}

	private static class ReplacerMap extends HashMap<String, Object> {
		public ReplacerMap(Map<? extends String, ? extends Object> map) {
			super(buildCompoundMap(map));
		}

		private static Map<String, ? extends Object> buildCompoundMap(Map<? extends String, ? extends Object> map) {
			HashMap<String, Object> compound = newHashMap();
			for (Map.Entry<? extends String, ? extends Object> e : map.entrySet()) {
				String key = e.getKey();
				Object value = e.getValue();
				decompose(compound, key, value);
			}
			return compound;
		}

		private static void decompose(HashMap<String, Object> compound, String key, Object value) {
			if (key.contains(".")) {
				int dotIndex = key.indexOf('.');
				String firstPart = key.substring(0, dotIndex);
				String secondPart = key.substring(dotIndex + 1);
				if (compound.containsKey(firstPart)) {
					Object o = compound.get(firstPart);
					if (o instanceof CompoundValueMap) {
						decompose((HashMap<String, Object>) o, secondPart, value);
					} else {
						CompoundValueMap map = new CompoundValueMap();
						map.setToStringValue(o);
						compound.put(firstPart, map);
						decompose(map, secondPart, value);
					}
				} else {
					HashMap<String, Object> map = new CompoundValueMap();
					decompose(map, secondPart, value);
					compound.put(firstPart, map);
				}
			} else {
				if (compound.containsKey(key)) {
					Object o = compound.get(key);
					if (o instanceof CompoundValueMap) {
						((CompoundValueMap) o).setToStringValue(value);
					} else {
						throw new RuntimeException("Cannot register same key twice");
					}
				} else {
					compound.put(key, value);
				}
			}
		}

		@Override
		public Object get(Object o) {
			Object s = super.get(o);
			if (IGNORE_PLACEHOLDER.equals(s)) {
				return "{{" + o + "}}";
			} else if (EMPTY_PLACEHOLDER.equals(s)) {
				return "";
			}
			return s;
		}
	}

	/**
	 * A map which has a toString() which can be set, and other values can be looked-up.
	 * For instance the case when you have "person" and "person.age" as keys.
	 */
	private static class CompoundValueMap extends HashMap<String, Object> {
		private Object toStringValue;
		public CompoundValueMap() {
			super();
		}

		public void setToStringValue(Object toStringValue) {
			this.toStringValue = toStringValue;
		}

		@Override
		public String toString() {
			return toStringValue != null ? toStringValue.toString() : super.toString();
		}
	}
}

