package com.xebialabs.deployit.service.deployment;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder;
import com.xebialabs.deployit.deployment.planner.Planner;
import com.xebialabs.deployit.exception.DeployitException;
import com.xebialabs.deployit.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.plugin.api.deployment.execution.Plan;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.deployment.InitialDeploymentTask;
import com.xebialabs.deployit.task.deployment.UndeploymentTask;
import com.xebialabs.deployit.task.deployment.UpgradeDeploymentTask;
import com.xebialabs.overthere.OverthereFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.InputStream;
import java.util.Collection;
import java.util.Set;

import static com.google.common.collect.Sets.newHashSet;

@Component
public class DeploymentService {

    private PojoConverter pojoConverter;

    private RepositoryService repositoryService;

    private Planner planner;


    @Autowired
    public DeploymentService(PojoConverter pojoConverter, RepositoryService repositoryService, Planner planner) {
        this.pojoConverter = pojoConverter;
        this.repositoryService = repositoryService;
        this.planner = planner;
    }

    public Task prepareInitialDeployment(final ConfigurationItemEntity pkgEntity, final ConfigurationItemEntity envEntity, final Collection<ConfigurationItemEntity> deployeds) {
        final PojoConverter.Context converter = pojoConverter.getContext();
        final DeploymentPackage pkg = converter.toPojo(pkgEntity);
        final Environment env = converter.toPojo(envEntity);

        DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().initial(pkg, env);

        Set<Deployed> deployedPojos = convertIncomingDeployeds(deployeds, converter);

        DeploymentOperationCalculator.calculate(builder, Sets.<Deployed>newHashSet(), deployedPojos);

        DeltaSpecification specification = builder.build();
        Plan plan = planner.plan(specification);
        return new InitialDeploymentTask(specification, plan.getSteps(), repositoryService, pojoConverter, converter);
    }

    public Task prepareUpgradeDeployment(final ConfigurationItemEntity newPkgEntity, final ConfigurationItemEntity existingDeploymentEntity, final Collection<ConfigurationItemEntity> deployeds) {
        final PojoConverter.Context oldConverterContext = pojoConverter.getContext();
        final DeployedApplication existingDeployment = oldConverterContext.toPojo(existingDeploymentEntity);

        final PojoConverter.Context newConverterContext = pojoConverter.getContext();
        final DeploymentPackage newPkg = newConverterContext.toPojo(newPkgEntity);

        final DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().upgrade(newPkg, existingDeployment);

        final Set<Deployed> incomingDeployedPojos = convertIncomingDeployeds(deployeds, newConverterContext);

        DeploymentOperationCalculator.calculate(builder, existingDeployment.getDeployeds(), incomingDeployedPojos);
        final DeltaSpecification specification = builder.build();
        final Plan plan = planner.plan(specification);
        return new UpgradeDeploymentTask(specification, existingDeployment, plan.getSteps(), repositoryService, pojoConverter, oldConverterContext, newConverterContext);
    }

    private Set<Deployed> convertIncomingDeployeds(Collection<ConfigurationItemEntity> deployeds, final PojoConverter.Context converter) {
        Collection<Deployed> pojos = converter.toPojo(deployeds);
        return newHashSet(pojos);
    }

    public Task prepareUndeployment(final ConfigurationItemEntity deployedApplicationEntity) {
        final PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
        final DeployedApplication deployedApplication = pojoConverterContext.toPojo(deployedApplicationEntity);

        DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().undeploy(deployedApplication);
        DeploymentOperationCalculator.calculate(builder, deployedApplication.getDeployeds(), Sets.<Deployed>newHashSet());
        DeltaSpecification specification = builder.build();
        Plan plan = planner.plan(specification);
        return new UndeploymentTask(specification, deployedApplication, plan.getSteps(), repositoryService, pojoConverter, pojoConverterContext);
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    public static class DeploymentException extends DeployitException {
        public DeploymentException(final String message) {
            super(message);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(DeploymentService.class);
}
