package com.xebialabs.deployit.service.deployment;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import org.springframework.stereotype.Component;

import java.util.*;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.service.replacement.MustachePlaceholderScanner.hasPlaceholders;
import static java.lang.String.format;

@Component
public class DeployedGenerator {

    public Deployed generateMostSpecificDeployed(Deployable d, Container c, ConsolidatedDictionary dictionary) {
        Type deployedType = findMostSpecificDeployedTypeForDeployableAndContainerTypes(d.getType(), c.getType());
        if (deployedType != null) {
            return generateDeployed(d, c, deployedType, dictionary);
        }
        return null;
    }

    public Deployed generateDeployed(Deployable d, Container c, Type deployedType, ConsolidatedDictionary dictionary) {
        Descriptor deployed = DescriptorRegistry.getDescriptor(deployedType);
        Deployed<Deployable, Container> configurationItem = deployed.newInstance();
        configurationItem.setDeployable(d);
        configurationItem.setContainer(c);

        IdGenerator.generateId(configurationItem);

        Descriptor deployableDescriptor = DescriptorRegistry.getDescriptor(d.getType());
        for (PropertyDescriptor propertyDescriptor : deployed.getPropertyDescriptors()) {
	        String name = propertyDescriptor.getName();

	        if (!propertyDescriptor.getName().equals("placeholders")) {
		        PropertyDescriptor deployablePropertyDescriptor = deployableDescriptor.getPropertyDescriptor(name);
			    if (deployablePropertyDescriptor != null) {
				    Object valueFromDeployable = deployablePropertyDescriptor.get(d);
				    setDeployedValue(valueFromDeployable, configurationItem, propertyDescriptor, dictionary);
			    }
	        } else {
		        // Placeholders are a special field, they should ALWAYS be present on both Deployable and Deployed
		        // They should only exist in (Deployed)Artifact, according to our Descriptor validation logic.
		        PropertyDescriptor deployableplaceholders = deployableDescriptor.getPropertyDescriptor("placeholders");
		        Set<String> placeholders = (Set<String>) deployableplaceholders.get(d);
		        propertyDescriptor.set(configurationItem, resolveAllPlaceholders(dictionary, Maps.<String, String>newHashMap(), placeholders));
	        }
        }
        return configurationItem;
    }

	private void setDeployedValue(Object value, Deployed<Deployable, Container> deployed, PropertyDescriptor property, ConsolidatedDictionary dictionary) {
		try {
			value = dictionary.resolve(value);
			property.set(deployed, value);
		} catch (DictionaryValueException dve) {
			logger.error("Could not resolve dictionary keys", dve);
		} catch (RuntimeException e) {
			logger.error("Could not convert (resolved) value to correct type", e);
		}
	}

	public Deployed generateUpgradeDeployed(Deployable newDeployable, Deployed deployed, ConsolidatedDictionary dictionary) {
        Descriptor deployedDesc = DescriptorRegistry.getDescriptor(deployed.getType());
        Type deployableType = deployedDesc.getDeployableType();
        Type newDeployableType = newDeployable.getType();
        checkArgument(deployableType.equals(newDeployableType) || DescriptorRegistry.getSubtypes(deployableType).contains(newDeployableType), "The new Deployable type %s should be assignable to %s", newDeployableType, deployableType);
        Deployed<Deployable, Container> newDeployed = deployedDesc.newInstance();

        Descriptor deployableDescriptor = DescriptorRegistry.getDescriptor(deployableType);
        for (PropertyDescriptor propertyDescriptor : deployedDesc.getPropertyDescriptors()) {
            String name = propertyDescriptor.getName();
	        if (name.equals("placeholders")) {
		        continue;
	        }
            Object oldDeployedValue = propertyDescriptor.get(deployed);
            Object valueToSet = oldDeployedValue;

            PropertyDescriptor deployablePropertyDescriptor = deployableDescriptor.getPropertyDescriptor(name);
            if (deployablePropertyDescriptor != null) {
	            Object oldDeployableValue = deployablePropertyDescriptor.get(deployed.getDeployable());
                if ((oldDeployableValue != null) && (oldDeployableValue.equals(oldDeployedValue)
		                || hasPlaceholders(oldDeployableValue.toString()))) {
                    valueToSet = deployablePropertyDescriptor.get(newDeployable);
                }

	            try {
		            valueToSet = dictionary.resolve(valueToSet);
	            } catch (DictionaryValueException dve) {
		            valueToSet = null;
		            logger.error("Could not resolve dictionary keys", dve);
	            }
            }
	        try {
                propertyDescriptor.set(newDeployed, valueToSet);
	        } catch (RuntimeException re) {
		        logger.error("Could not convert (resolved) value to correct type", re);
	        }
        }

		copyOverPlaceholdersToDeployed(newDeployed, newDeployable, deployed, dictionary);

        // Overwrite the Deployable with the new deployable
        newDeployed.setDeployable(newDeployable);
        newDeployed.setId(deployed.getId());

        return newDeployed;
    }

	private void copyOverPlaceholdersToDeployed(Deployed<Deployable, Container> newDeployed, Deployable newDeployable, Deployed deployed, ConsolidatedDictionary dictionary) {
		Descriptor deployedDesc = DescriptorRegistry.getDescriptor(newDeployed.getType());
		Descriptor deployableDesc = DescriptorRegistry.getDescriptor(newDeployable.getType());

		PropertyDescriptor deployedPlaceholdersDesc = deployedDesc.getPropertyDescriptor("placeholders");
		if (deployedPlaceholdersDesc != null) {
			Map<String, String> existingDeployedPlaceholders = (Map<String, String>) deployedPlaceholdersDesc.get(deployed);
			Set<String> newDeployablePlaceholders = (Set<String>) deployableDesc.getPropertyDescriptor("placeholders").get(newDeployable);

			deployedPlaceholdersDesc.set(newDeployed, resolveAllPlaceholders(dictionary, existingDeployedPlaceholders, newDeployablePlaceholders));
		}
	}

	private Map<String, String> resolveAllPlaceholders(ConsolidatedDictionary dictionary, Map<String, String> existingDeployedPlaceholders, Set<String> newDeployablePlaceholders) {
		Map<String, String> newDeployedPlaceholders = newHashMap();
		for (String newDeployablePlaceholder : newDeployablePlaceholders) {
			if (existingDeployedPlaceholders.containsKey(newDeployablePlaceholder)) {
				newDeployedPlaceholders.put(newDeployablePlaceholder, existingDeployedPlaceholders.get(newDeployablePlaceholder));
			} else {
				String resolved = "";
				try {
					resolved = (String) dictionary.resolve("{{" + newDeployablePlaceholder + "}}");
				} catch (DictionaryValueException e) {
					// Ok.
				}
				newDeployedPlaceholders.put(newDeployablePlaceholder, resolved);
			}
		}
		return newDeployedPlaceholders;
	}

	public Type findMostSpecificDeployedTypeForDeployableAndContainerTypes(Type deployableType, Type containerType) {
        List<Type> deployedTypesForDeployableAndContainer = findDeployedTypesForDeployableAndContainerTypes(deployableType, containerType);

        if (deployedTypesForDeployableAndContainer.isEmpty()) {
            return null;
        }
        if (deployedTypesForDeployableAndContainer.size() == 1) {
            return deployedTypesForDeployableAndContainer.get(0);
        }

        List<Type> copy = filterMostSpecific(deployedTypesForDeployableAndContainer);

        if (copy.size() == 1) {
            return copy.get(0);
        }

        throw new IllegalArgumentException(format("Found %s for %s and %s, expected only 1", copy, deployableType, containerType));
    }

    List<Type> filterMostSpecific(List<Type> deployedTypesForDeployableAndContainer) {
        List<Type> copy = newArrayList(deployedTypesForDeployableAndContainer);

        for (Type currentType : deployedTypesForDeployableAndContainer) {
            Descriptor currentDescriptor = DescriptorRegistry.getDescriptor(currentType);

            for (Iterator<Type> copyIt = copy.iterator(); copyIt.hasNext();) {
                Type comparedType = copyIt.next();
                Descriptor comparedDescriptor = DescriptorRegistry.getDescriptor(comparedType);

                boolean deployableEquals = comparedDescriptor.getDeployableType().equals(currentDescriptor.getDeployableType());
                boolean containerEquals = comparedDescriptor.getContainerType().equals(currentDescriptor.getContainerType());
                boolean containerSupertype = isSubType(comparedDescriptor.getContainerType(), currentDescriptor.getContainerType());
                boolean deployableSupertype = isSubType(comparedDescriptor.getDeployableType(), currentDescriptor.getDeployableType());

                boolean isSupertype = isSubType(comparedType, currentType);

                if ((containerSupertype && deployableEquals)
                        || (deployableSupertype && containerEquals)
                        || (deployableSupertype && containerSupertype)
                        || isSupertype) {
                    copyIt.remove();
                }
            }
        }
        return copy;
    }

    private boolean isSubType(Type superType, Type subType) {
        return DescriptorRegistry.getSubtypes(superType).contains(subType);
    }

    public List<Type> findDeployedTypesForDeployableAndContainerTypes(Type deployableType, Type containerType) {
        Collection<Type> deployedTypes = DescriptorRegistry.getSubtypes(Type.valueOf(Deployed.class));
        final Descriptor deployableDescriptor = DescriptorRegistry.getDescriptor(deployableType);
        final Descriptor containerDescriptor = DescriptorRegistry.getDescriptor(containerType);
        return newArrayList(Collections2.filter(deployedTypes, new Predicate<Type>() {
            @Override
            public boolean apply(Type input) {
                Descriptor deployedDescriptor = DescriptorRegistry.getDescriptor(input);
                return !deployedDescriptor.isVirtual() && deployableDescriptor.isAssignableTo(deployedDescriptor.getDeployableType()) && containerDescriptor.isAssignableTo(deployedDescriptor.getContainerType());
            }
        }));
    }

	private static final Logger logger = LoggerFactory.getLogger(DeployedGenerator.class);
}
