package com.xebialabs.deployit.security.permission;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.security.permission.PermissionHandler.Action.DENY;
import static com.xebialabs.deployit.security.permission.PermissionHandler.Action.GRANT;
import static com.xebialabs.deployit.security.permission.PermissionHandler.Action.REVOKE;
import static com.xebialabs.deployit.security.permission.PermissionHelper.grantPermission;
import static com.xebialabs.deployit.security.permission.PermissionHelper.revokePermission;
import static com.xebialabs.deployit.security.permission.PrivilegesHelper.denyPrivileges;
import static com.xebialabs.deployit.security.permission.PrivilegesHelper.grantPrivileges;
import static com.xebialabs.deployit.security.permission.PrivilegesHelper.revokePrivileges;

import java.util.Collection;
import java.util.List;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.repository.JcrRepositoryServiceHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class PermissionHandler {

	protected static final String JCR_READ = "jcr:read";
	protected static final String JCR_ADD_CHILD_NODES = "jcr:addChildNodes";
	protected static final String JCR_NODE_TYPE_MANAGEMENT = "jcr:nodeTypeManagement";
	protected static final String JCR_VERSION_MANAGEMENT = "jcr:versionManagement";
	protected static final String JCR_MODIFY_PROPERTIES = "jcr:modifyProperties";
	protected static final String JCR_REMOVE_CHILD_NODES = "jcr:removeChildNodes";
	protected static final String JCR_REMOVE_NODE = "jcr:removeNode";
	protected static final List<String> ADD_CHILD_NODES_PRIVILEGES = newArrayList(JCR_READ, JCR_ADD_CHILD_NODES, JCR_NODE_TYPE_MANAGEMENT, JCR_VERSION_MANAGEMENT, JCR_MODIFY_PROPERTIES);
	/**
	 * Contents:
	 * - JCR_READ: You must be able to read in order to delete
	 * - JCR_REMOVE_CHILD_NODES: The ability to unlink child nodes from this node
	 * - JCR_REMOVE_NODE: The ability to remove a (child) node
	 * - JCR_VERSION_MANAGEMENT: Because we create a new version on delete
	 * - JCR_MODIFY_PROPERTIES: Because we need to be able to delete properties from the (child) nodes.
	 */
	protected static final List<String> REMOVE_CHILD_NODES_PRIVILEGES = newArrayList(JCR_READ, JCR_REMOVE_CHILD_NODES, JCR_REMOVE_NODE, JCR_VERSION_MANAGEMENT, JCR_MODIFY_PROPERTIES);

	enum Action { GRANT, REVOKE, DENY }

	public final void grant(String user, List<String> onConfigurationItems) {
		if (onConfigurationItems.isEmpty()) {
			handle(user, GRANT);
		} else {
			for (String onConfigurationItem : onConfigurationItems) {
				if (JcrRepositoryServiceHolder.getRepositoryService().checkNodeExists(onConfigurationItem)) {
					handle(user, onConfigurationItem, GRANT);
				} else {
					logger.warn("Skipping grant of {} to {} on {}", new Object[] {getPermission(), user, onConfigurationItem});
				}
			}
		}
	}

	protected void handle(String user, Action action) {
		throw new Checks.IncorrectArgumentException("Cannot %s permission %s without any nodes", action.name(), getPermission().getPermissionName());
	}
	protected void handle(String user, String onConfigurationItem, Action action) {
		throw new Checks.IncorrectArgumentException("Cannot %s permission %s on nodes, it is a global permission", action.name(), getPermission().getPermissionName());
	}

	public final void revoke(String user, List<String> onConfigurationItems) {
		if (onConfigurationItems.isEmpty()) {
			if (PermissionHelper.hasPermission(getPermission())) {
				handle(user, REVOKE);
			}
		} else {
			for (String onConfigurationItem : onConfigurationItems) {
				if (PermissionHelper.hasPermission(getPermission(), user, onConfigurationItem)) {
					handle(user, onConfigurationItem, REVOKE);
				}
			}
		}
	}

	public void deny(String user, List<String> onConfigurationItems) {
		throw new Checks.IncorrectArgumentException("Cannot deny %s, only grant/revoke supported.", getPermission());
	}

	abstract Permission getPermission();

	public final boolean hasPermission(List<String> onConfigurationItems) {
		return PermissionHelper.hasPermission(getPermission(), onConfigurationItems);
	}

	protected final void permission(Permission permission, String principal, Action action) {
		if (action == GRANT) {
			grantPermission(permission, principal);
		} else if (action == REVOKE) {
			revokePermission(permission, principal);
		}
	}

	protected final void permission(Permission permission, String principal, String onConfigurationItem, Action action) {
		if (action == GRANT) {
			grantPermission(permission, principal, onConfigurationItem);
		} else if (action == REVOKE) {
			revokePermission(permission, principal, onConfigurationItem);
		}
	}

	protected final void limitedPrivileges(List<String> privileges, String principal, String onConfigurationItem, Action action) {
		doPrivileges(privileges, principal, onConfigurationItem, action, false);
	}
	protected final void privileges(List<String> privileges, String principal, String onConfigurationItem, Action action) {
		doPrivileges(privileges, principal, onConfigurationItem, action, true);
	}

	private void doPrivileges(List<String> privileges, String principal, String onConfigurationItem, Action action, boolean transitive) {
		if (!JcrRepositoryServiceHolder.getRepositoryService().checkNodeExists(onConfigurationItem)) {
			return;
		}

		if (action == GRANT) {
			grantPrivileges(privileges, principal, onConfigurationItem, transitive);
		} else if (action == REVOKE) {
			revokePrivileges(privileges, principal, onConfigurationItem, transitive);
		} else if (action == DENY) {
			denyPrivileges(privileges, principal, onConfigurationItem, transitive);
		}
	}

	protected final void limitedPrivileges(List<String> privileges, String principal, Collection<String> onConfigurationItems, Action action) {
		for (String onConfigurationItem : onConfigurationItems) {
			limitedPrivileges(privileges, principal, onConfigurationItem, action);
		}
	}
	protected final void privileges(List<String> privileges, String principal, Collection<String> onConfigurationItems, Action action) {
		for (String onConfigurationItem : onConfigurationItems) {
			privileges(privileges, principal, onConfigurationItem, action);
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(PermissionHandler.class);
}
