package com.xebialabs.deployit.repository;

import static com.google.common.base.Joiner.on;
import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.jcr.JcrConstants.ARTIFACT_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.NESTED;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.ReferentialIntegrityException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.nodetype.NodeType;
import javax.jcr.query.Query;
import javax.jcr.query.QueryResult;
import javax.jcr.query.RowIterator;
import javax.jcr.version.VersionManager;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;

@Component
public class JcrRepositoryService implements RepositoryService {

    private JcrTemplate jcrTemplate;

    @Autowired
    public JcrRepositoryService(JcrTemplate jcrTemplate) {
        this.jcrTemplate = jcrTemplate;
    }

    @Override
    public void execute(ChangeSet changeset) {
        jcrTemplate.execute(new ChangeSetExecutor(changeset));
    }

    @Override
    public void checkReferentialIntegrity(final ChangeSet changeset) throws ItemInUseException, ItemAlreadyExistsException {
        jcrTemplate.execute(new ReferentialIntegrityChecker(changeset));
    }

    @Override
    public <T extends RepositoryObjectEntity> void create(final T... entities) {
        ChangeSet changeset = new ChangeSet();
        changeset.setCreateEntities(Lists.<RepositoryObjectEntity>newArrayList(entities));
        execute(changeset);
    }

    @Override
    public <T extends RepositoryObjectEntity> T read(final String id) {
        checkNotNull(id, "id is null");

        return this.<T>read(id, false);
    }

    @Override
    public <T extends RepositoryObjectEntity> T readFully(String id) {
        checkNotNull(id, "id is null");

        return this.<T>read(id, true);
    }

    private <T extends RepositoryObjectEntity> T read(final String id, final boolean fully) {
        checkNotNull(id, "id is null");

        return jcrTemplate.execute(new JcrCallback<T>() {
            @Override
            public T doInJcr(Session session) throws IOException, RepositoryException {
                try {
                    Node node = session.getNode(getAbsolutePathFromId(id));
                    return new EntityNodeReader(session, node, fully).<T>getEntity();
                } catch (PathNotFoundException exc) {
                    throw new NotFoundException(exc, "Repository entity [%s] not found", id);
                }
            }
        });
    }

    @Override
    public <T extends RepositoryObjectEntity> void update(final T... entities) {
        ChangeSet changeset = new ChangeSet();
        changeset.setUpdateEntities(Lists.<RepositoryObjectEntity>newArrayList(entities));
        execute(changeset);
    }

    @Override
    public void delete(final String... ids) {
        ChangeSet changeset = new ChangeSet();
        changeset.setDeleteEntityIds(Lists.<String>newArrayList(ids));
        execute(changeset);
    }

    @Override
    public List<String> list(final SearchParameters parameters) {
        checkNotNull(parameters);

        List<RepositoryObjectEntity> entities = listEntities(parameters);
        List<String> entityIds = Lists.newArrayList();
        for (RepositoryObjectEntity each : entities) {
            entityIds.add(each.getId());
        }
        return entityIds;
    }

    @Override
    public <T extends RepositoryObjectEntity> List<T> listEntities(final SearchParameters parameters) {
        checkNotNull(parameters, "parameters is null");
        return jcrTemplate.execute(new JcrCallback<List<T>>() {
            @Override
            public List<T> doInJcr(final Session session) throws IOException, RepositoryException {
                final Query query = new SearchQueryBuilder(parameters).build(session);
                final QueryResult queryResult = query.execute();
                if(parameters.at != null) {
                	return getDistinctEntities(parameters, session, queryResult);
                } else {
                	return getOrderedEntities(parameters, session, queryResult);
                }
            }
        });
    }

    private <T extends RepositoryObjectEntity> List<T> getDistinctEntities(final SearchParameters parameters, final Session session, final QueryResult queryResult) throws RepositoryException, IOException {
	    Map<String, T> entities = new LinkedHashMap<String, T>();
	    final RowIterator iterator = queryResult.getRows();
	    while (iterator.hasNext()) {
	        // TODO temporary fix!
	        // We need to fix the security model for deployment to
	        // inherit the privileges from both the source and the
	        // target.
	        // Remove the try-catch once that is implemented.
	        try {
	            Node each = iterator.nextRow().getNode(SearchQueryBuilder.CI_SELECTOR_NAME);
	            T entity = new EntityNodeReader(session, each, false).<T>getEntity();
	            if (entityNotFoundYetOrNewerThenEntityAlreadyFound(entities, entity)) {
	                entities.put(entity.getId(), entity);
	            }
	        } catch (RepositoryException rre) {
	            // Ignore: We weren't allowed to read a node or one of it's relations.
	        }
	    }
	    return new ArrayList<T>(entities.values());
    }

    private <T extends RepositoryObjectEntity> boolean entityNotFoundYetOrNewerThenEntityAlreadyFound(final Map<String, T> entities, final T entity) {
        return !entities.containsKey(entity.getId())
                || entities.get(entity.getId()).getLastModified().before(entity.getLastModified());
    }

	private <T extends RepositoryObjectEntity> List<T> getOrderedEntities(final SearchParameters parameters, final Session session, final QueryResult queryResult) throws RepositoryException, IOException {
		List<T> entities = newArrayList();
	    final RowIterator iterator = queryResult.getRows();
	    while (iterator.hasNext()) {
	        // TODO temporary fix!
	        // We need to fix the security model for deployment to
	        // inherit the privileges from both the source and the
	        // target.
	        // Remove the try-catch once that is implemented.
	        try {
	            Node each = iterator.nextRow().getNode(SearchQueryBuilder.CI_SELECTOR_NAME);
	            entities.add(new EntityNodeReader(session, each, false).<T>getEntity());
	        } catch (RepositoryException rre) {
	            // Ignore: We weren't allowed to read a node or one of it's relations.
	        }
	    }

		// Because JCR always returns the current node before any versioned node (while we want it last) and because reverse sorting by the last-modified-date
		// property does not seem to work, we have to implement sorting by last-modified-date in Java. This does not increase memory usage, but will be slower
		// when a lot of entities are returned. Luckily this only happens when specifying an after or a before condition because sorting by ID _is_ handled by
		// JCR.
	    if(parameters.after != null || parameters.before != null) {
		    Collections.sort(entities, new Comparator<T>() {
	            public int compare(T lhs, T rhs) {
	            	int diff = lhs.getId().compareTo(rhs.getId());
	            	if(diff != 0) {
	            		return diff;
	            	} else {
	            		return lhs.getLastModified().compareTo(rhs.getLastModified());
	            	}
	            }
			});
	    }

	    return entities;
    }

    @Override
    public boolean checkNodeExists(final String id) {
        return jcrTemplate.execute(new JcrCallback<Boolean>() {
            @Override
            public Boolean doInJcr(final Session session) throws IOException, RepositoryException {
                return session.itemExists(getAbsolutePathFromId(id));
            }
        });
    }

	static String generateJcrPropertyNameForNestedProperty(PropertyDescriptor pd, PropertyDescriptor nestedPd) {
        return pd.getName() + "_" + nestedPd.getName();
    }

    private static class ChangeSetExecutor implements JcrCallback<Object> {

        private final ChangeSet changeset;

        ChangeSetExecutor(ChangeSet changeset) {
            checkNotNull(changeset);
            this.changeset = changeset;
        }

        @Override
        public Object doInJcr(Session session) throws IOException, RepositoryException {
            execute(session, true);
            return null;
        }

		public void execute(Session session, boolean autocommit) throws RepositoryException {
			try {
				createEntities(changeset.getCreateEntities(), session);
				updateEntities(changeset.getUpdateEntities(), session);
				deleteEntities(changeset.getDeleteEntityIds(), session);
				if (autocommit) {
					session.save();
				}
			} catch (ReferentialIntegrityException exc) {
				throw new ItemInUseException(exc, "Cannot delete configuration items [%s] because one of the configuration items, or one of their children, is still being referenced", on(',').join(changeset.getDeleteEntityIds()));
			}
		}

        private void createEntities(List<RepositoryObjectEntity> entities, Session session) throws RepositoryException {
            verifyEntitiesMeetCreationPreconditions(entities);
            Collections.sort(entities, new EntitiesSortHierarchyComparator());
            List<EntityNodeWriter> writers = newArrayList();
            for (RepositoryObjectEntity entity : entities) {
                Node node = createNode(entity, session);
                final EntityNodeWriter nodeWriter = new EntityNodeWriter(session, entity, node);
                nodeWriter.writeBasics();
                writers.add(nodeWriter);
            }
            writeEntities(writers);
        }

        private void updateEntities(List<RepositoryObjectEntity> entities, Session session) throws RepositoryException {
            List<Node> nodes = retrieveAndCheckpointNodesToBeUpdated(entities, session);
            for (int i = 0; i < entities.size(); i++) {
                updateNode(entities.get(i), nodes.get(i), session);
            }
        }

        private void deleteEntities(List<String> entityIds, Session session) throws RepositoryException {
            List<Node> nodes = retrieveAndCheckpointNodesToBeDeleted(entityIds, session);
            for (Node node : nodes) {
                deleteNode(node);
            }
        }

        private List<Node> retrieveAndCheckpointNodesToBeUpdated(List<RepositoryObjectEntity> entities, Session session) throws RepositoryException {
            List<Node> nodes = newArrayList();
            for (RepositoryObjectEntity entity : entities) {
                checkNotNull(entity, "entity is null");
                checkNotNull(entity.getId(), "id is null");
                try {
                    final Node node = session.getNode(getAbsolutePathFromId(entity.getId()));
                    checkpoint(node, session);
                    nodes.add(node);
                } catch (PathNotFoundException exc) {
                    throw new NotFoundException("Repository entity [%s] not found", entity.getId());
                }
            }
            return nodes;
        }

        private List<Node> retrieveAndCheckpointNodesToBeDeleted(List<String> ids, Session session) throws RepositoryException {
            List<Node> nodes = newArrayList();
            for (String id : ids) {
                checkNotNull(id, "id is null");
                try {
                    Node node = session.getNode(getAbsolutePathFromId(id));
                    checkpoint(node, session);
                    nodes.add(node);
                } catch (PathNotFoundException ignored) {
                    //ignore
                }
            }
            return nodes;
        }

        private Node createNode(RepositoryObjectEntity entity, Session session) throws RepositoryException {
            validateNodeStoredInCorrectPath(entity, session);
            final Node node = session.getRootNode().addNode(entity.getId());
            int index = node.getIndex();
            if (index != 1) {
                node.remove();
                throw new ItemAlreadyExistsException("Repository entity with ID [%s] already exists", entity.getId());
            }
            setMixins(entity, node);
            return node;
        }

        private void updateNode(RepositoryObjectEntity entity, Node node, Session session) throws RepositoryException {
            new EntityNodeWriter(session, entity, node).write();
        }

        private void deleteNode(Node node) throws RepositoryException {
            node.remove();
        }

        private void checkpoint(Node node, Session session) throws RepositoryException {
            VersionManager versionManager = session.getWorkspace().getVersionManager();
            versionManager.checkpoint(node.getPath());
        }

        private void verifyEntitiesMeetCreationPreconditions(List<RepositoryObjectEntity> entities) {
            for (int i = 0; i < entities.size(); i++) {
                RepositoryObjectEntity entity = entities.get(i);
                checkNotNull(entity, "entity at index %s is null.", i);
                checkNotNull(entity.getId(), "entity at index %s has null id.", i);

                if (entity instanceof ArtifactEntity) {
                    checkArgument(((ArtifactEntity) entity).getFilename() != null, "Artifact %s should have filename", entity.getId());
                    checkArgument(((ArtifactEntity) entity).containsData(), "Artifact %s should contain data", entity.getId());
                }
            }
        }

        private void validateNodeStoredInCorrectPath(RepositoryObjectEntity entity, Session session) throws RepositoryException {
            final String id = entity.getId();
            final Type type = entity.getType();
            final Descriptor desc = DescriptorRegistry.getDescriptor(type);
            checkNotNull(desc, "Unknown configuration item type %s", type);

            final String[] pathElements = id.split("/");
            if (desc.getRoot() != NESTED) {
                Checks.checkArgument(pathElements.length == 2 && pathElements[0].equals(desc.getRoot().getRootNodeName()),
                        "Configuration item of type %s cannot be stored at %s. It should be stored under %s", type, id, desc.getRoot().getRootNodeName());
            } else {
                Checks.checkArgument(pathElements.length >= 3,
                        "Configuration item of type %s cannot be stored at %s. It should be stored under a valid parent node", type, id);
                final String parentId = id.substring(0, id.lastIndexOf('/'));
                final Node parentNode = session.getNode(getAbsolutePathFromId(parentId));
                Checks.checkArgument(parentNode != null, "Configuration item of type %s cannot be stored at %s. The parent does not exist", type, id);
                Checks.checkArgument(parentNode.isNodeType(CONFIGURATION_ITEM_NODETYPE_NAME),
                        "Configuration item of type %s cannot be stored at %s. The parent is not a configuration item", type, id);
                final String parentTypeName = parentNode.getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString();
                final Descriptor parentDescriptor = DescriptorRegistry.getDescriptor(parentTypeName);

                // Check parent relation
                for (PropertyDescriptor aPD : desc.getPropertyDescriptors()) {
                    if (aPD.getKind() == CI && aPD.isAsContainment()) {
                        if (matchesType(parentDescriptor, aPD.getReferencedType())) {
                            return;
                        }
                    }
                }

                // Check child relation
                for (PropertyDescriptor aPD : parentDescriptor.getPropertyDescriptors()) {
                    if (aPD.getKind() == SET_OF_CI && aPD.isAsContainment()) {
                        if (matchesType(desc, aPD.getReferencedType())) {
                            return;
                        }
                    }
                }

                throw new Checks.IncorrectArgumentException(
                        "Configuration item of type %s cannot be stored at %s. The parent cannot contain configuration items of this type", type, id);
            }
        }

        private boolean matchesType(final Descriptor desc, final Type wantedType) {
            // FIXME: Use Descriptor.isAssignableTo(Type) and inline
            return desc.getType().equals(wantedType) || DescriptorRegistry.getSubtypes(wantedType).contains(desc.getType());
        }

        private void setMixins(RepositoryObjectEntity entity, Node node)
                throws RepositoryException {
            if (entity instanceof ArtifactEntity) {
                node.addMixin(ARTIFACT_NODETYPE_NAME);
                node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
            } else if (entity instanceof ConfigurationItemEntity) {
                node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
            } else {
                throw new IllegalArgumentException("Not supporting RepositoryEntity type " + entity.getClass().getName());
            }
            node.addMixin(NodeType.MIX_REFERENCEABLE);
            node.addMixin(NodeType.MIX_VERSIONABLE);
        }

        private void writeEntities(List<EntityNodeWriter> writers) throws RepositoryException {
            for (EntityNodeWriter writer : writers) {
                writer.write();
            }
        }
    }

    private static class EntitiesSortHierarchyComparator implements Comparator<RepositoryObjectEntity> {
        @Override
        public int compare(RepositoryObjectEntity o1, RepositoryObjectEntity o2) {
            final int nrSlashes1 = StringUtils.countOccurrencesOf(o1.getId(), "/");
            final int nrSlashes2 = StringUtils.countOccurrencesOf(o2.getId(), "/");
            return nrSlashes1 - nrSlashes2;
        }
    }

    private static class ReferentialIntegrityChecker implements JcrCallback<Object> {

        private final ChangeSet changeset;

        ReferentialIntegrityChecker(ChangeSet changeset) {
            checkNotNull(changeset);
            this.changeset = changeset;
        }

        @Override
        public Object doInJcr(Session session) throws IOException, RepositoryException {
            checkIfAnyOfTheEntitiesToBeCreatedAlreadyExists(session);
            Multimap<String, PropertyRef> deletedEntityRefs = findReferencesToDeletedNodes(session);
            ChangeSetExecutor changeSetExecutor = new ChangeSetExecutor(changeset);
            changeSetExecutor.execute(session, false);
            try {
                checkForRemainingReferencesToDeletedNodes(deletedEntityRefs, session);
            } finally {
                session.refresh(false);
            }
            return null;
        }

        private void checkIfAnyOfTheEntitiesToBeCreatedAlreadyExists(Session session) throws RepositoryException {
            for (RepositoryObjectEntity entity : changeset.getCreateEntities()) {
                if (session.nodeExists(getAbsolutePathFromId(entity.getId()))) {
                    throw new ItemAlreadyExistsException("Repository entity %s already exists", entity.getId());
                }
            }
        }

        private Multimap<String, PropertyRef> findReferencesToDeletedNodes(Session session) throws RepositoryException {
            Multimap<String, PropertyRef> deletedEntityRefs = ArrayListMultimap.create();
            for (String deleteNodeId : changeset.getDeleteEntityIds()) {
                if (!session.nodeExists(getAbsolutePathFromId(deleteNodeId))) {
                    continue;
                }
                Node node = session.getNode(getAbsolutePathFromId(deleteNodeId));
                PropertyIterator references = node.getReferences();
                while (references.hasNext()) {
                    Property property = references.nextProperty();
                    Node parent = property.getParent();
                    deletedEntityRefs.put(deleteNodeId, new PropertyRef(parent.getPath(), property.getName(), node.getIdentifier()));
                }
            }
            return deletedEntityRefs;
        }

        private void checkForRemainingReferencesToDeletedNodes(Multimap<String, PropertyRef> deletedEntityRefs, Session session) throws RepositoryException {
            for (String deleteNodeId : deletedEntityRefs.keySet()) {
                for (PropertyRef propertyRef : deletedEntityRefs.get(deleteNodeId)) {
                    if (session.nodeExists(propertyRef.owningNode)) {
                        Node owningNode = session.getNode(propertyRef.owningNode);
                        if (owningNode.hasProperty(propertyRef.property)) {
                            Property property = owningNode.getProperty(propertyRef.property);
                            Value[] values = property.isMultiple() ? property.getValues() : new Value[]{property.getValue()};
                            for (Value value : values) {
                                if (value.getString().equals(propertyRef.uuidOfReferencedNode)) {
                                    throw new ItemInUseException("Repository entity %s is still referenced by %s", deleteNodeId, propertyRef.owningNode);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    private static class PropertyRef {
        private String owningNode;
        private String property;
        private String uuidOfReferencedNode;

        PropertyRef(String owningNode, String property, String uuidOfReferencedNode) {
            this.owningNode = owningNode;
            this.property = property;
            this.uuidOfReferencedNode = uuidOfReferencedNode;
        }
    }

}
