package com.xebialabs.deployit.repository;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.io.Closeables.closeQuietly;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CREATING_TASK_ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.DATA_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.FILENAME_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_DATE_PROPERTY_NAME;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;

import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;

import org.apache.jackrabbit.value.ReferenceValue;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;

/**
 * Writes an {@link RepositoryObjectEntity} to a JCR {@link Node}.
 */
class EntityNodeWriter {

	private final Session session;
    private final RepositoryObjectEntity entity;
	private final Node node;
	private boolean basicsWritten = false;

	public EntityNodeWriter(final Session session, final RepositoryObjectEntity entity, final Node node) {
		this.session = session;
		this.entity = entity;
		this.node = node;
	}

	public void writeBasics() throws RepositoryException {
		copyConfigurationItemTypeName();
		basicsWritten = true;
	}

	public void write() throws RepositoryException {
		if (!basicsWritten) {
			writeBasics();
		}
		copyData();
		copyValuesIntoNode();
		copyMetadata();
	}

	private void copyConfigurationItemTypeName() throws RepositoryException {
		node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, entity.getType().toString());
	}

	private void copyData() throws RepositoryException {
		if (entity instanceof ArtifactEntity) {
			ArtifactEntity artifact = (ArtifactEntity) entity;
			if(artifact.getFilename() != null) {
				node.setProperty(FILENAME_PROPERTY_NAME, artifact.getFilename());
			}
			if (artifact.containsData()) {
				try {
					InputStream dataInput = artifact.getData().getInput();
					try {
						Binary binary = session.getValueFactory().createBinary(dataInput);
						node.setProperty(DATA_PROPERTY_NAME, binary);
					} finally {
						closeQuietly(dataInput);
					}
				} catch(IOException exc) {
					throw new RepositoryException("Cannot read artifact data for " + artifact, exc);
				}
			}
		}
	}

	private void copyValuesIntoNode() throws RepositoryException {
		Descriptor ciDescriptor = DescriptorRegistry.getDescriptor(entity.getType());
		for (PropertyDescriptor pd : ciDescriptor.getPropertyDescriptors()) {
			if (entity.getValue(pd.getName()) == null) {
				removePropertyFromNode(pd);
				continue;
			}

			if(pd.isHidden()) {
				continue;
			}

			switch (pd.getKind()) {
			case BOOLEAN:
			case INTEGER:
			case STRING:
			case ENUM:
				copyPrimitivePropertyIntoNode(pd);
				break;
			case SET_OF_STRING:
				copySetOfStringsPropertyIntoNode(pd);
				break;
			case CI:
				copyConfigurationItemPropertyIntoNode(pd);
				break;
			case SET_OF_CI:
				copySetOfConfigurationItemsPropertyIntoNode(pd);
				break;
			case MAP_STRING_STRING:
				copyMapPropertyIntoNode(pd);
				break;
			default:
				throw new IllegalArgumentException("Cannot convert property " + pd.getName() + " because it is of unsupported kind " + pd.getKind());

			}
		}
	}

	@SuppressWarnings("unchecked")
    private void copyMapPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
		Map<String, String> value = (Map<String, String>) entity.getValue(pd.getName());
		List<Value> valueList = newArrayList();
		for (String s : value.keySet()) {
			Value v = session.getValueFactory().createValue(s + "=" + value.get(s));
			valueList.add(v);
		}

		Value[] jcrValues = valueList.toArray(new Value[valueList.size()]);
		node.setProperty(pd.getName(), jcrValues);
	}

	private void copyMetadata() throws RepositoryException {
		node.setProperty(ID_PROPERTY_NAME, entity.getId());
		if (entity.getOverrideLastModified() != null) {
			node.setProperty(LAST_MODIFIED_DATE_PROPERTY_NAME, entity.getOverrideLastModified());
		} else {
			node.setProperty(LAST_MODIFIED_DATE_PROPERTY_NAME, Calendar.getInstance());
		}
		if (entity.getCreatingTaskId() != null) {
			node.setProperty(CREATING_TASK_ID_PROPERTY_NAME, entity.getCreatingTaskId());
		}
	}

	private void removePropertyFromNode(PropertyDescriptor pd) throws RepositoryException {
		try {
			node.getProperty(pd.getName()).remove();
		} catch (PathNotFoundException ignored) {
		}
	}

	private void copyPrimitivePropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
		String valueAsString = convertObjectValueToString(pd, entity.getValue(pd.getName()));
		node.setProperty(pd.getName(), valueAsString);
	}

	private void copySetOfStringsPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
		final ValueFactory valueFactory = session.getValueFactory();

		Collection<?> valueAsCollection = convertObjectValueToCollection(pd, entity.getValue(pd.getName()));
		List<Value> jcrValueList = newArrayList();
		for (Object each : valueAsCollection) {
			if (!(each instanceof String)) {
				throw new IllegalArgumentException("Element in property " + pd.getName() + " of repository entity " + entity.getId()
				        + " is not a String: " + each);
			}
			String stringValue = (String) each;
			jcrValueList.add(valueFactory.createValue(stringValue));
		}
		Value[] jcrValues = jcrValueList.toArray(new Value[jcrValueList.size()]);
		node.setProperty(pd.getName(), jcrValues);
	}

	private void copyConfigurationItemPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
		if (pd.isAsContainment()) {
			return;
		}

		String referencedCiId = convertObjectValueToString(pd, entity.getValue(pd.getName()));
		Node referencedCi = session.getNode(getAbsolutePathFromId(referencedCiId));
		node.setProperty(pd.getName(), referencedCi);
	}

	private void copySetOfConfigurationItemsPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
		if (pd.isAsContainment()) {
			return;
		}

		Collection<?> valueAsCollection = convertObjectValueToCollection(pd, entity.getValue(pd.getName()));
		List<Value> jcrReferenceList = newArrayList();
		for (Object each : valueAsCollection) {
			if (!(each instanceof String)) {
				throw new IllegalArgumentException("Element in property " + pd.getName() + " of repository entity " + entity.getId()
				        + " is not a String: " + each);
			}
			String referencedCiId = (String) each;
			Node referencedCi = session.getNode(getAbsolutePathFromId(referencedCiId));
			jcrReferenceList.add(new ReferenceValue(referencedCi));
		}
		Value[] jcrReferenceValues = jcrReferenceList.toArray(new Value[jcrReferenceList.size()]);
		node.setProperty(pd.getName(), jcrReferenceValues);
	}

	private String convertObjectValueToString(PropertyDescriptor pd, Object value) {
		if (!(value instanceof String)) {
			throw new IllegalArgumentException("Property " + pd.getName() + " of repository entity " + entity.getId() + " is not a String: " + value);
		}
		return (String) value;
	}

	private Collection<?> convertObjectValueToCollection(PropertyDescriptor pd, Object value) {
		if (!(value instanceof Collection)) {
			throw new IllegalArgumentException("Property " + pd.getName() + " of repository entity " + entity.getId() + " is not a Collection: " + value);
		}
		return (Collection<?>) value;
	}
}
