/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.typedescriptor;

import static com.google.common.base.Preconditions.checkNotNull;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.xebialabs.deployit.reflect.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.ListMultimap;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.DeployItConfiguration;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;

/**
 * Container for all configuration item descriptors.
 */
@Component("descriptorRepository")
public class ConfigurationItemTypeDescriptorRepository implements ConfigurationItemDescriptorFactoryHelper {

	private Map<String, ConfigurationItemDescriptor> descriptorsByType = new HashMap<String, ConfigurationItemDescriptor>();

	private ListMultimap<String, ConfigurationItemDescriptor> descriptorsBySuperType = LinkedListMultimap.create();

	@PostConstruct
	public void initialize() {
		new ConfigurationItemDescriptorFactory(this);
	}

	/**
	 * Returns all descriptors.
	 * 
	 * @return an unsorted collection of all descriptors.
	 */
	public Collection<ConfigurationItemDescriptor> getDescriptors() {
		return descriptorsByType.values();
	}

	/**
	 * Returns a map of all descriptors.
	 * 
	 * @return a map that maps a type name to its descriptor.
	 */
	public Map<String, ConfigurationItemDescriptor> getDescriptorMap() {
		return Collections.unmodifiableMap(descriptorsByType);
	}

	public ConfigurationItemDescriptor getDescriptorByType(final String type) {
		return descriptorsByType.get(type);
	}

	public ConfigurationItemDescriptor getDescriptorByShortName(final String shortName) {
		return (ConfigurationItemDescriptor) CollectionUtils.find(descriptorsByType.values(), new Predicate() {
			@Override
			public boolean evaluate(Object descriptor) {
				return StringUtils.equalsIgnoreCase(((ConfigurationItemDescriptor) descriptor).getSimpleName(), shortName);
			}
		});
	}

	public ConfigurationItemDescriptor getDescriptorByClass(final Class<?> clazz) {
		return descriptorsByType.get(clazz.getName());
	}

	public List<ConfigurationItemDescriptor> getDescriptorsBySuperType(final String supertype) {
		return descriptorsBySuperType.get(supertype);
	}

	@Deprecated
	@Override
	public ConfigurationItemDescriptor getDescriptor(final Serializable object) {
		return getDescriptorForObject(object);
	}

	public ConfigurationItemDescriptor getDescriptorForObject(final Serializable object) {
		Serializable ci = object;
		if (ConfigurationItemProxy.class.isInstance(object)) {
			ci = ((ConfigurationItemProxy) object).getBackingObject();
		}
		return descriptorsByType.get(ci.getClass().getName());
	}

	public void loadConfigurationItems(Collection<Class<?>> configurationItemTypes, List<String> loadingErrors) {
		descriptorsByType.clear();

		DeployItConfiguration config = DeployItConfiguration.getInstance();

		for (Class<?> type : checkNotNull(configurationItemTypes)) {
			ConfigurationItemDescriptor descriptor;
			if (type.isAnnotationPresent(ConfigurationItem.class)) {
				descriptor = new ConfigurationItemDescriptor(type, config.getDefaults());
				for (String each : descriptor.getSuperClasses()) {
					descriptorsBySuperType.put(each, descriptor);
				}
				for (String each : descriptor.getInterfaces()) {
					descriptorsBySuperType.put(each, descriptor);
				}
				descriptorsBySuperType.put(descriptor.getType(), descriptor);
				descriptorsByType.put(descriptor.getType(), descriptor);
			} else {
				String errorMessage = type.getName() + " does not have an @" + ConfigurationItem.class.getSimpleName() + " annotation. Ignoring it.";
				logger.error(errorMessage);
				loadingErrors.add(errorMessage);
			}
		}
	}

	public boolean isSimilar(Serializable ci1, Serializable ci2) {
		return ConfigurationItemReflectionUtils.isSimilar(ci1, ci2);
	}

	public boolean isIdentical(Serializable ci1, Serializable ci2) {
		return ConfigurationItemReflectionUtils.isIdentical(ci1, ci2);
	}

	private static final Logger logger = LoggerFactory.getLogger(ConfigurationItemTypeDescriptorRepository.class);
}
