package com.xebialabs.deployit.translation;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.BaseConfigurationItem;
import com.xebialabs.deployit.MappingInfo;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.DeploymentPackage;
import com.xebialabs.deployit.ci.Environment;
import com.xebialabs.deployit.ci.MiddlewareResource;
import com.xebialabs.deployit.ci.artifact.DeployableArtifact;
import com.xebialabs.deployit.ci.mapping.KeyValuePair;
import com.xebialabs.deployit.ci.mapping.Mapping;
import com.xebialabs.deployit.exception.DeployitException;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyType;
import com.xebialabs.deployit.reflect.ConfigurationItemReflectionUtils;
import com.xebialabs.deployit.service.deployment.DeploymentService;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemTypeDescriptorRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.ReflectionUtils;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static org.springframework.beans.BeanUtils.copyProperties;
import static org.springframework.beans.BeanUtils.instantiate;

@Component
public class MappingGenerator {
    @Autowired
    private ConfigurationItemTypeDescriptorRepository repository;

    private static Field labelField;

    private static Field sourceField;

    private static Field targetField;

    static {
        try {
            labelField = BaseConfigurationItem.class.getDeclaredField("label");
            labelField.setAccessible(true);
            sourceField = Mapping.class.getDeclaredField("source");
            sourceField.setAccessible(true);
            targetField = Mapping.class.getDeclaredField("target");
            targetField.setAccessible(true);
        } catch (SecurityException e) {
            throw new Error(e);
        } catch (NoSuchFieldException e) {
            throw new Error(e);
        }
    }

    public List<MappingInfo> generateMappingsForInitialDeployment(final DeploymentPackage pkg, final Environment env, final String mappingType) {
        Deployment deployment = new Deployment(pkg, env);
        deployment.setLabel(DeploymentService.generateDeploymentLabel(pkg, env));

        List<MappingInfo> initialMappings = newArrayList();
        for (Serializable eachPkgMember : pkg.getAllMembers()) {
            final List<MappingInfo> mappingsInfo = createInitialMappings(deployment, eachPkgMember, mappingType);
            initialMappings.addAll(mappingsInfo);
        }
        return initialMappings;
    }

    public List<MappingInfo> generateMappingsForUpgradeDeployment(Deployment d, DeploymentPackage newPkg) {
        List<MappingInfo> generatedMappingsForUpgrade = newArrayList();
        for (Mapping<?, ?> each : d.getMappings()) {
            MappingInfo generatedMappingForUpgrade = upgradeMapping(each, d, newPkg);
            if (generatedMappingForUpgrade != null) {
                generatedMappingsForUpgrade.add(generatedMappingForUpgrade);
            }
        }
        return generatedMappingsForUpgrade;
    }

    private List<MappingInfo> createInitialMappings(Deployment deployment, Serializable pkgMember, String mappingType) {
        List<MappingInfo> mappingsCreated = newArrayList();
        ConfigurationItemDescriptor descriptor;
        for (Serializable eachEnvMember : deployment.getTarget().getMembers()) {
            if (mappingType == null) {
                descriptor = findFirstMatchingMappingDescriptor(pkgMember, eachEnvMember);
            } else {
                descriptor = getDescriptorIfMappingIsCompatible(pkgMember, eachEnvMember, mappingType);
            }
            if (descriptor != null) {
                mappingsCreated.add(new MappingInfo(createMapping(deployment, pkgMember, eachEnvMember, descriptor), true));
            }
        }
        return mappingsCreated;
    }

    private ConfigurationItemDescriptor findFirstMatchingMappingDescriptor(Serializable pkgMember, Serializable envMember) {
        List<String> allPkgMemberTypes = getAllTypes(pkgMember.getClass());
        List<String> allEnvMemberTypes = getAllTypes(envMember.getClass());
        for (ConfigurationItemDescriptor eachDescriptor : repository.getDescriptors()) {
            if (!isMappingDescriptor(eachDescriptor))
                continue;
            if (!allPkgMemberTypes.contains(eachDescriptor.getMappingSourceType()))
                continue;
            if (!allEnvMemberTypes.contains(eachDescriptor.getMappingTargetType()))
                continue;

            return eachDescriptor;
        }
        return null;
    }

    private ConfigurationItemDescriptor getDescriptorIfMappingIsCompatible(Serializable pkgMember, Serializable envMember, String mappingType) {
        ConfigurationItemDescriptor descriptor = repository.getDescriptorByShortName(mappingType);
        List<String> allPkgMemberTypes = getAllTypes(pkgMember.getClass());
        List<String> allEnvMemberTypes = getAllTypes(envMember.getClass());
        if (descriptor != null && isMappingDescriptor(descriptor) && allPkgMemberTypes.contains(descriptor.getMappingSourceType()) && allEnvMemberTypes.contains(descriptor.getMappingTargetType())) {
            return descriptor;
        }
        return null;
    }

    private Mapping<?, ?> createMapping(Deployment deployment, Serializable pkgMember, Serializable envMember, ConfigurationItemDescriptor descriptor) {
        Mapping<?, ?> mapping = (Mapping<?, ?>) descriptor.newInstance();
        String mappingLabel = generateMappingLabel(pkgMember, envMember);
        ReflectionUtils.setField(labelField, mapping, mappingLabel);
        ReflectionUtils.setField(sourceField, mapping, pkgMember);
        ReflectionUtils.setField(targetField, mapping, envMember);
        if (descriptor.getPlaceholdersField() != null) {
            ReflectionUtils.setField(descriptor.getPlaceholdersField(), mapping, generatePlaceholders(pkgMember));
        }
        mapping.postInit(deployment);
        return mapping;
    }

    @SuppressWarnings("unchecked")
    private List<KeyValuePair> generatePlaceholders(final Serializable pkgMember) {
        final ConfigurationItemDescriptor descriptor = repository.getDescriptorForObject(pkgMember);
        final ConfigurationItemPropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor("placeholders");
        List<KeyValuePair> placeholders = new ArrayList<KeyValuePair>();
        if (propertyDescriptor != null) {
            final Set<String> placeholderSet = (Set<String>) propertyDescriptor.getPropertyValueFromConfigurationItem(pkgMember);
            for (String placeholder : placeholderSet) {
                placeholders.add(new KeyValuePair(placeholder, null));
            }
        }

        return placeholders;
    }

    private MappingInfo upgradeMapping(Mapping<?, ?> mapping, Deployment d, DeploymentPackage pkg) {
        Serializable source = mapping.getSource();
        List<Serializable> similarObjects = findSimilarObject(pkg, source);
        if (similarObjects.size() == 0) {
            // Could not find a similar object, but we want to return the mapping without a source set.
            return new MappingInfo(cloneMappingForUpgrade(mapping, source, pkg, d.getSource()), false);
        } else if (similarObjects.size() == 1) {
            return new MappingInfo(cloneMappingForUpgrade(mapping, similarObjects.get(0), pkg, d.getSource()), true);
        } else {
            throw new DeployitException("Mapping " + mapping
                    + " could not be upgraded since multiple similar items were found with the same identifying property value");
        }
    }

    private Mapping<?, ?> cloneMappingForUpgrade(Mapping<?, ?> mapping, Serializable newSource, DeploymentPackage newPkg, DeploymentPackage oldPkg) {
        @SuppressWarnings("unchecked")
        Mapping<Serializable, Serializable> clonedMapping = instantiate(mapping.getClass());
        final ConfigurationItemDescriptor descriptor = repository.getDescriptorByClass(mapping.getClass());
        copyProperties(mapping, clonedMapping);
        clonedMapping.setSource(newSource);
        clonedMapping.setLabel(generateMappingLabel(newSource, clonedMapping.getTarget()));
        cloneAndReplacePlaceholderFieldInMapping(newSource, clonedMapping, descriptor);
        replaceCiReferencesInMappingToReferToSimilarObjectsInNewDeploymentPackage(clonedMapping, descriptor, newPkg, oldPkg);
        return clonedMapping;
    }

    private void cloneAndReplacePlaceholderFieldInMapping(Serializable newSource,
                                                          Mapping<Serializable, Serializable> mapping, ConfigurationItemDescriptor descriptor) {
        final Field placeholdersField = descriptor.getPlaceholdersField();
        if (placeholdersField != null) {
            try {
                // This code was added to support manually added placeholders due to them not being detected during import.
                // Once we've got a better detection mechanism in place, you should not be able to add manual placeholders,
                // so we can revert commit: 3cb453d3431952f27279013c531ef77b95fb153d (FIXME)
                final List<KeyValuePair> newPlaceholders = generatePlaceholders(newSource);
                @SuppressWarnings("unchecked")
                final List<KeyValuePair> oldPlaceholders = (List<KeyValuePair>) placeholdersField.get(mapping);
                for (KeyValuePair oldPlaceholder : oldPlaceholders) {
                    final KeyValuePair existingPlaceholder = getByKey(oldPlaceholder.getKey(), newPlaceholders);
                    if (existingPlaceholder != null) {
                        existingPlaceholder.setValue(oldPlaceholder.getValue());
                    }
                    else{
                        newPlaceholders.add(oldPlaceholder);
                    }
                }
                placeholdersField.set(mapping, newPlaceholders);
            } catch (IllegalAccessException e) {
                throw new IllegalStateException("Ask Jeroen", e);
            }
        }
    }

    private void replaceCiReferencesInMappingToReferToSimilarObjectsInNewDeploymentPackage(
            Mapping<?, ?> mapping, ConfigurationItemDescriptor descriptor,
            DeploymentPackage newPkg, DeploymentPackage oldPkg) {

        ConfigurationItemPropertyDescriptor[] propertyDescriptors = descriptor.getPropertyDescriptors();

        for (ConfigurationItemPropertyDescriptor propertyDescriptor : propertyDescriptors) {
            if (propertyDescriptor.getName().equals("source") || propertyDescriptor.getName().equals("target")) {
                continue;
            }

            if (propertyDescriptor.getType() == ConfigurationItemPropertyType.CI) {
                replaceCiPropertyInMappingWithSimilarObjectInNewPackage(mapping, newPkg, oldPkg, propertyDescriptor);
            } else if (propertyDescriptor.getType() == ConfigurationItemPropertyType.SET_OF_CIS) {
                replaceSetOfCisPropertyInMappingWithSimilarObjectsInNewPackage(mapping, newPkg, oldPkg, propertyDescriptor);
            }
        }
    }

    private void replaceCiPropertyInMappingWithSimilarObjectInNewPackage(
            Mapping<?, ?> mapping, DeploymentPackage newPkg, DeploymentPackage oldPkg, ConfigurationItemPropertyDescriptor propertyDescriptor) {

        Object mappingPropertyValue = propertyDescriptor.getPropertyValueFromConfigurationItem(mapping);
        if (thereIsAnIdenticalObjectInOldDeploymentPackage(oldPkg, mappingPropertyValue)) {
            Serializable similarObjectInNewPkg = findSimilarObject(newPkg, mappingPropertyValue, mapping, propertyDescriptor.getName());
            propertyDescriptor.setPropertyValueInConfigurationItem(mapping, similarObjectInNewPkg);
        }
    }

    private boolean thereIsAnIdenticalObjectInOldDeploymentPackage(DeploymentPackage pkg, Object serializableSource) {
        Serializable source = (Serializable) serializableSource;
        for (MiddlewareResource middlewareResource : pkg.getMiddlewareResources()) {
            if (ConfigurationItemReflectionUtils.isIdentical(middlewareResource, source)) {
                return true;
            }
        }
        for (DeployableArtifact each : pkg.getDeployableArtifacts()) {
            if (ConfigurationItemReflectionUtils.isIdentical(each, source)) {
                return true;
            }
        }
        return false;
    }

    private void replaceSetOfCisPropertyInMappingWithSimilarObjectsInNewPackage(
            Mapping<?, ?> mapping, DeploymentPackage newPkg, DeploymentPackage oldPkg, ConfigurationItemPropertyDescriptor propertyDescriptor) {

        Set<Serializable> setOfCis = (Set<Serializable>) propertyDescriptor.getPropertyValueFromConfigurationItem(mapping);
        if (setOfCis == null) {
            return;
        }

        Set<Serializable> newSetOfCis = Sets.newHashSet();
        for (Serializable ci : setOfCis) {
            if (thereIsAnIdenticalObjectInOldDeploymentPackage(oldPkg, ci)) {
                Serializable similarObjectInNewPkg = findSimilarObject(newPkg, ci, mapping, propertyDescriptor.getName());
                if (similarObjectInNewPkg != null) {
                    newSetOfCis.add(similarObjectInNewPkg);
                }
            } else {
                newSetOfCis.add(ci);
            }
        }

        propertyDescriptor.setPropertyValueInConfigurationItem(mapping, newSetOfCis);
    }

    private Serializable findSimilarObject(DeploymentPackage pkg, Object source, Mapping<?, ?> mapping, String mappingPropertyName) {
        List<Serializable> similarObjects = findSimilarObject(pkg, (Serializable) source);
        if (similarObjects.isEmpty()) {
            return null;
        }

        if (similarObjects.size() == 1) {
            return similarObjects.get(0);
        }

        throw new DeployitException("Mapping " + mapping
                + " could not be upgraded since multiple similar items were found with the same identifying property value for mapping property "
                + mappingPropertyName);
    }

    private List<Serializable> findSimilarObject(DeploymentPackage pkg, Serializable source) {
        List<Serializable> similarObjects = newArrayList();
        for (MiddlewareResource each : pkg.getMiddlewareResources()) {
            if (ConfigurationItemReflectionUtils.isSimilar(each, source)) {
                similarObjects.add(each);
            }
        }
        for (DeployableArtifact each : pkg.getDeployableArtifacts()) {
            if (ConfigurationItemReflectionUtils.isSimilar(each, source)) {
                similarObjects.add(each);
            }
        }
        return similarObjects;
    }

    private String generateMappingLabel(Serializable pkgMember, Serializable envMember) {
        final String environmentMemberLabel = repository.getDescriptorForObject(envMember).getLabelValueFromConfigurationItem(envMember);
        return environmentMemberLabel + "/" + getIdentifier(pkgMember);
    }

    private String getIdentifier(final Serializable object) {
        StringBuilder builder = new StringBuilder();
        final ConfigurationItemDescriptor descriptor = repository.getDescriptorForObject(object);
        for (ConfigurationItemPropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {
            if (propertyDescriptor.isIdentifying()) {
                Object value = propertyDescriptor.getPropertyValueFromConfigurationItem(object);
                if (value != null) {
                    if (builder.length() > 0) {
                        builder.append(":");
                    }
                    builder.append(value.toString().replace('/', '_'));
                }
            }
        }

        if (builder.length() == 0) {
            final String ciLabel = descriptor.getLabelValueFromConfigurationItem(object);
            builder.append(ciLabel.substring(ciLabel.lastIndexOf("/") + 1));
        }

        return builder.toString();
    }

    private boolean isMappingDescriptor(ConfigurationItemDescriptor eachDescriptor) {
        return eachDescriptor.getSuperClasses().contains(Mapping.class.getName());
    }

    private List<String> getAllTypes(Class<?> clazz) {
        ConfigurationItemDescriptor desc = repository.getDescriptorByClass(clazz);
        checkNotNull(desc, "Cannnot find descriptor for " + clazz);
        return getAllTypesForDescriptor(desc);
    }

    private List<String> getAllTypesForDescriptor(ConfigurationItemDescriptor descriptor) {
        checkNotNull(descriptor);

        List<String> types = Lists.newArrayList();
        types.add(descriptor.getType());
        types.addAll(descriptor.getSuperClasses());
        types.addAll(descriptor.getInterfacesWithoutSerializable());
        return types;
    }

    private KeyValuePair getByKey(String key, List<KeyValuePair> placeHolders) {
        for (KeyValuePair keyValuePair : placeHolders) {
            if (keyValuePair.getKey().equals(key)) {
                return keyValuePair;
            }
        }
        return null;
    }

}
