package com.xebialabs.deployit.translation;

import static com.xebialabs.deployit.ConfigurationItemRoot.NESTED;
import static com.xebialabs.deployit.repository.JcrPathHelper.buildId;

import java.io.Serializable;
import java.util.Collection;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.ConfigurationItemRoot;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemTypeDescriptorRepository;

@Component
public class DiscoveryIdGenerator {

	@Autowired
	private ConfigurationItemTypeDescriptorRepository repository;

	/**
	 * Generate the correct JCR ids for the discovered CIs
	 * 
	 * Basically we're going to loop over all the CIs, and then:
	 * 
	 * <pre>
	 * 1) Check whether the CI is rooted, then we prefix the id with the root
	 * 2) If not rooted, find the CI property that is modeled asContainment, and check the parent is present in the correct ids set
	 * 2a) If it is present, prefix my id with the parent
	 * 2b) If not, throw an {@link IllegalStateException}
	 * </pre>
	 * 
	 * <b>N.B. 1:</b> This method assumes that a parent come earlier in the list then its children
	 * 
	 * <b>N.B. 2:</b> SET_OF_CI properties are not taken into account
	 * 
	 * @param discoveredItems
	 *            the discovered CIs
	 */
	public void generateCorrectIds(Collection<Serializable> discoveredItems) {
		for (Serializable each : discoveredItems) {
			generateCorrectId(each);
		}
	}

	private void generateCorrectId(Serializable discoveredItem) {
		ConfigurationItemDescriptor descriptor = repository.getDescriptorForObject(discoveredItem);
		if (descriptor == null) {
			throw new IllegalStateException("Cannot find descriptor for object of class " + discoveredItem.getClass().getName());
		}

		String discoveredItemId = descriptor.getLabelValueFromConfigurationItem(discoveredItem);
		if (discoveredItemId.contains("/")) {
			return;
		}

		ConfigurationItemRoot root = descriptor.getRoot();
		if (root != NESTED) {
			generateIdBasedOnRoot(discoveredItem, descriptor, discoveredItemId, root);
			return;
		}

		for (ConfigurationItemPropertyDescriptor eachPropertyDescriptor : descriptor.getPropertyDescriptors()) {
			if (eachPropertyDescriptor.asContainment()) {
				generateIdBasedOnParentId(discoveredItem, descriptor, discoveredItemId, eachPropertyDescriptor);
				return;
			}
		}

		throw new IllegalArgumentException("The discovered item " + discoveredItemId + " is not rooted and does not have a parent");
	}

	private void generateIdBasedOnRoot(Serializable discoveredItem, ConfigurationItemDescriptor descriptor, String discoveredItemId, ConfigurationItemRoot root) {
		String generatedId = buildId(root, discoveredItemId);
		descriptor.setLabelValueInConfigurationItem(discoveredItem, generatedId);
	}

	private void generateIdBasedOnParentId(Serializable discoveredItem, ConfigurationItemDescriptor descriptor, String discoveredItemId,
	        ConfigurationItemPropertyDescriptor eachPropertyDescriptor) {
		Object parent = getParent(discoveredItem, discoveredItemId, eachPropertyDescriptor);
		String parentId = getParentId(discoveredItemId, parent);
		String generatedId = buildId(parentId, discoveredItemId);
		descriptor.setLabelValueInConfigurationItem(discoveredItem, generatedId);
	}

	private Object getParent(Serializable discoveredItem, String discoveredItemId, ConfigurationItemPropertyDescriptor eachPropertyDescriptor) {
		Object parent = eachPropertyDescriptor.getPropertyValueFromConfigurationItem(discoveredItem);
		if (parent == null) {
			throw new IllegalArgumentException("The discovered item " + discoveredItemId + " is not rooted and its parent is null");
		}
		if (!(parent instanceof Serializable)) {
			throw new IllegalArgumentException("The discovered item " + discoveredItemId + " is not rooted and its parent is not a Serializable");
		}
		return parent;
	}

	private String getParentId(String discoveredItemId, Object parent) {
		ConfigurationItemDescriptor parentDescriptor = repository.getDescriptorForObject((Serializable) parent);
		String parentId = parentDescriptor.getLabelValueFromConfigurationItem(parent);
		if (parentId == null) {
			throw new IllegalArgumentException("The discovered item " + discoveredItemId + " is not rooted and its parent does not have an id");
		}
		if (!(parentId.contains("/"))) {
			throw new IllegalArgumentException("The discovered item " + discoveredItemId + " is not rooted and its parent's id does not contain a slash");
		}
		return parentId;
	}

}
